import React, {useMemo} from 'react';
import {useLocation, useParams} from 'react-router-dom';
import {useSelector} from 'react-redux';

import {EProjectName} from 'constants/common';

import {EFooterProject} from 'components/Footer/types';
import {ELoadableReducer} from 'types/common/ELoadableReducer';

import flightReducer from 'reducers/avia/flight/reducer';
import contextReducer from 'reducers/avia/context/reducer';

import nonceSelector from 'selectors/common/nonceSelector';

import useServerDataFetcher from 'utilities/hooks/useServerDataFetcher';
import prefetchData from 'projects/avia/pages/FlightPage/utilities/prefetchData';
import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import getFlightQuery from 'projects/avia/pages/FlightPage/utilities/getFlightQuery';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/aviaFlight';

import withReducers from 'containers/withReducers/withReducers';

import LayoutDefault from 'components/Layouts/LayoutDefault/LayoutDefault';
import Breadcrumbs from 'projects/avia/pages/FlightPage/components/Breadcrumbs/Breadcrumbs';
import Container from 'components/Layouts/Container/Container';
import Dates from 'projects/avia/pages/FlightPage/components/Dates/Dates';
import Map from 'projects/avia/pages/FlightPage/components/Map/Map';
import Segments from 'projects/avia/pages/FlightPage/components/Segments/Segments';
import {TwoColumnLayout} from 'components/Layouts/TwoColumnLayout/TwoColumnLayout';
import UsefulLinks from 'projects/avia/pages/FlightPage/components/UsefulLinks/UsefulLinks';
import Book from 'projects/avia/pages/FlightPage/components/Book/Book';
import MainInfo from 'projects/avia/pages/FlightPage/components/MainInfo/MainInfo';
import SeoText from 'projects/avia/pages/FlightPage/components/SeoText/SeoText';
import Direct from 'projects/avia/pages/FlightPage/components/Direct/Direct';
import Helmet from 'projects/avia/pages/FlightPage/components/Helmet/Helmet';
import Heading from 'components/Heading/Heading';
import Text from 'components/Text/Text';
import LayoutError500 from 'components/Layouts/LayoutError500/LayoutError500';
import BookLoader from 'components/BookLoader/BookLoader';
import NotFoundContent from 'components/NotFound/NotFoundContent/NotFoundContent';
import NotFoundWrapper from 'components/NotFound/NotFoundWrapper/NotFoundWrapper';
import {IErrorActionButton} from 'components/ErrorModal/ErrorModal';

import useRequestFlight from 'projects/avia/pages/FlightPage/hooks/useRequestFlight';
import useUpdateContext from 'projects/avia/pages/FlightPage/hooks/useUpdateContext';
import useResetOrder from 'projects/avia/pages/FlightPage/hooks/useResetOrder';

import cx from './FlightPage.scss';

const ERROR_ACTION: IErrorActionButton = {
    title: i18nBlock.fetchErrorButton(),
    type: 'button',
    handler: (): void => {
        document.location.reload();
    },
};

const ROOT_QA = 'flightPage';

const FlightPage: React.FC = () => {
    const deviceType = useDeviceType();
    const location = useLocation();

    const {iataAndFlightNumber} = useParams<{iataAndFlightNumber: string}>();

    const [iata, flightNumber] = iataAndFlightNumber.split('-');

    const {when, from: fromCode} = getFlightQuery(location);

    const flightInfo = useRequestFlight({iata, flightNumber, when});

    const needToRender = useServerDataFetcher([prefetchData]);

    const nonce = useSelector(nonceSelector);

    useUpdateContext(when);
    useResetOrder();

    const contentBlock = useMemo(() => {
        if (flightInfo.isFailed) {
            if (flightInfo.isNotFound) {
                return (
                    <NotFoundWrapper>
                        <NotFoundContent />
                    </NotFoundWrapper>
                );
            }

            return (
                <LayoutError500
                    title={i18nBlock.fetchErrorTitle()}
                    subtitle={i18nBlock.fetchErrorSubtitle()}
                    action={ERROR_ACTION}
                />
            );
        }

        if (!flightInfo.isFetched) {
            return <BookLoader isLoading />;
        }

        const {blocks, seoInfo} = flightInfo.value;

        const activeDepartureStationIndex = fromCode
            ? blocks.flight.segments.findIndex(
                  ({departure}) => departure.stationCode === fromCode,
              )
            : 0;

        const mapBlock = (
            <div className={cx('mapOverlay')}>
                <Map
                    airportsCoords={blocks.map.flightPointCoordinates}
                    activeDepartureStationIndex={activeDepartureStationIndex}
                />
            </div>
        );

        const disclaimerBlock = (
            <Text
                className={cx('disclaimer')}
                size={deviceType.isMobile ? 's' : 'xs'}
                color="secondary"
                tag="div"
            >
                {blocks.disclaimer.text}
            </Text>
        );

        const segmentsBlock = (
            <Segments
                className={cx('segments')}
                items={blocks.flight.segments}
                activeFromStationCode={fromCode}
                mapBlock={mapBlock}
                disclaimerBlock={disclaimerBlock}
            />
        );

        const usefulLinksBlock = (
            <UsefulLinks
                title={blocks.usefulLinks.title}
                items={blocks.usefulLinks.items}
            />
        );

        const seoTextBlock = (
            <SeoText
                className={cx('seoText')}
                title={blocks.flightSeoText.title}
                description={blocks.flightSeoText.description}
            />
        );

        return (
            <div>
                <Helmet
                    iata={iata}
                    flightNumber={flightNumber}
                    when={when}
                    fromStationCode={fromCode}
                    seoInfo={seoInfo}
                    breadcrumbs={blocks.breadCrumbs.items}
                    nonce={nonce}
                />

                <Container>
                    <Breadcrumbs items={blocks.breadCrumbs.items} />

                    <Heading
                        className={cx('title')}
                        level={1}
                        {...prepareQaAttributes({
                            parent: ROOT_QA,
                            current: 'title',
                        })}
                    >
                        {blocks.flightTitle.title}
                    </Heading>

                    <Dates
                        className={cx('dates')}
                        items={blocks.flightDates.items}
                    />
                </Container>

                {deviceType.isMobile ? (
                    <Container>{segmentsBlock}</Container>
                ) : (
                    <div className={cx('mapAndSegmentsWrapper')}>
                        {mapBlock}

                        <Container className={cx('segmentsWrapper')}>
                            {segmentsBlock}
                        </Container>
                    </div>
                )}

                <Container>
                    {deviceType.isDesktop && disclaimerBlock}

                    <TwoColumnLayout
                        className={cx('columns')}
                        rightColumnOffset={10}
                        rightColumnWidth={80}
                        deviceType={deviceType}
                    >
                        <TwoColumnLayout.LeftColumn>
                            <MainInfo
                                className={cx('mainInfo')}
                                tariffs={blocks.additionalInfo.tariffs}
                                baggage={blocks.additionalInfo.baggage}
                                stats={blocks.additionalInfo.flightRating}
                                aircraftType={
                                    blocks.additionalInfo.aircraftType
                                }
                                description={blocks.additionalInfo.seoText}
                            />

                            {deviceType.isDesktop && usefulLinksBlock}

                            {deviceType.isDesktop && seoTextBlock}
                        </TwoColumnLayout.LeftColumn>

                        <TwoColumnLayout.RightColumn>
                            {blocks.tickets && (
                                <Book
                                    text={blocks.tickets.text}
                                    button={blocks.tickets.button}
                                    {...prepareQaAttributes({
                                        parent: ROOT_QA,
                                        current: 'book',
                                    })}
                                />
                            )}

                            <Direct className={cx('direct')} />

                            {deviceType.isMobile && usefulLinksBlock}

                            {deviceType.isMobile && seoTextBlock}
                        </TwoColumnLayout.RightColumn>
                    </TwoColumnLayout>
                </Container>
            </div>
        );
    }, [
        deviceType,
        flightInfo.isFailed,
        flightInfo.isFetched,
        flightInfo.isNotFound,
        flightInfo.value,
        flightNumber,
        fromCode,
        iata,
        nonce,
        when,
    ]);

    if (!needToRender) {
        return null;
    }

    return (
        <LayoutDefault
            className={cx('root', deviceMods('root', deviceType))}
            footerClassName={cx('footer')}
            project={EProjectName.AVIA}
            footerType={EFooterProject.AVIA}
            hasSideSheetNavigation
            isFixedNavigation
            showSearchForm
            showNavigation
        >
            {contentBlock}
        </LayoutDefault>
    );
};

export default withReducers([
    [ELoadableReducer.AVIA_FLIGHT, flightReducer],
    [ELoadableReducer.AVIA_CONTEXT, contextReducer],
])(FlightPage);
