import React from 'react';

import {ISeoInfo} from 'types/hotels/common/ISeoInfo';
import {
    IAviaBreadCrumb,
    IAviaFlightSchemaOrgInfo,
} from 'server/api/AviaGatewayApi/types/IAviaGetFlightPageApiResponse';

import getFlightPageUrl from 'projects/avia/lib/urls/getFlightPageUrl';
import {internalUrl} from 'utilities/url';

import DocumentMeta from 'components/DocumentMeta/DocumentMeta';
import SocialSharingMeta from 'components/SocialSharingMeta/SocialSharingMeta';
import SchemaMarkup, {
    getMetaBreadcrumbList,
} from 'components/SchemaMarkup/SchemaMarkup';

interface IHelmetProps {
    iata: string;
    flightNumber: string;
    when: string | undefined;
    fromStationCode: string | undefined;
    seoInfo: ISeoInfo<IAviaFlightSchemaOrgInfo>;
    breadcrumbs: IAviaBreadCrumb[];
    nonce: string;
}

const Helmet: React.FC<IHelmetProps> = props => {
    const {
        seoInfo: {
            title,
            description,
            openGraph: {
                title: ogTitle,
                description: ogDescription,
                image: ogImage,
                imageSize: ogImageSize,
            },
        },
        iata,
        flightNumber,
        when,
        fromStationCode,
        breadcrumbs,
        nonce,
    } = props;

    const canonicalUrl = getFlightPageUrl({
        iata,
        number: flightNumber,
        options: {
            withOrigin: true,
        },
    });

    const socialUrl = getFlightPageUrl({
        iata,
        number: flightNumber,
        when,
        fromStationCode,
        options: {
            withOrigin: true,
        },
    });

    const iataAndFlightNumber = `${iata}-${flightNumber}`;

    const alternates = [
        {
            href: canonicalUrl,
            hreflang: 'ru',
        },
        {
            href: `https://avia.yandex.ua/flights/${iataAndFlightNumber}/`,
            hreflang: 'uk',
        },
        {
            href: `https://avia.yandex.ua/flights/${iataAndFlightNumber}/?lang=ru`,
            hreflang: 'ru-UA',
        },
        {
            href: `https://bilet.yandex.com.tr/flights/${iataAndFlightNumber}/`,
            hreflang: 'tr',
        },
        {
            href: `https://avia.yandex.kz/flights/${iataAndFlightNumber}/`,
            hreflang: 'ru-KZ',
        },
    ];

    const breadcrumbsSchema = getMetaBreadcrumbList(
        breadcrumbs.map(({title: breadcrumbTitle, link: breadcrumbLink}) => ({
            name: breadcrumbTitle,
            url: breadcrumbLink
                ? internalUrl(breadcrumbLink, null, {withOrigin: true})
                : socialUrl,
        })),
    );

    return (
        <>
            <DocumentMeta
                title={title}
                description={description}
                canonicalUrl={canonicalUrl}
                alternates={alternates}
            />

            <SocialSharingMeta
                url={socialUrl}
                title={ogTitle}
                description={ogDescription}
                image={ogImage}
                imageHeight={ogImageSize?.height ?? ''}
                imageWidth={ogImageSize?.width ?? ''}
                twitterCardType="summary_large_image"
            />

            <SchemaMarkup data={breadcrumbsSchema} nonce={nonce} />
        </>
    );
};

export default React.memo(Helmet);
