import React, {useMemo, useState} from 'react';
import {IYMapsApi} from 'react-yandex-maps';
import {renderToString} from 'react-dom/server';

import {EColors} from 'projects/avia/constants/constants';

import {IAviaCoordinates} from 'server/api/AviaGatewayApi/types/IAviaGetFlightPageApiResponse';
import {IWithClassName} from 'types/withClassName';

import {useDeviceType} from 'utilities/hooks/useDeviceType';

import YandexMapsComponent, {
    GeoObject,
    YandexMapPlacemark,
} from 'components/YandexMaps/YandexMaps';

import cx from './Map.scss';

interface IMapProps extends IWithClassName {
    airportsCoords: IAviaCoordinates[];
    activeDepartureStationIndex: number;
}

/**
 * Карта с точками-аэропортами перелёта
 * @see react-yandex-maps https://react-yandex-maps.vercel.app/
 */
const Map: React.FC<IMapProps> = props => {
    const {className, airportsCoords, activeDepartureStationIndex} = props;

    const [mapInstance, setMapInstance] = useState<IYMapsApi>();

    const deviceType = useDeviceType();

    const [topLeftCornerCoodrs, bottomRightCornerCoodrs] = useMemo(() => {
        const latitudes = airportsCoords.map(({latitude}) => latitude);
        const longitudes = airportsCoords.map(({longitude}) => longitude);

        const minLatitude = Math.min(...latitudes);
        const maxLatitude = Math.max(...latitudes);
        const minLongitude = Math.min(...longitudes);
        const maxLongitude = Math.max(...longitudes);

        const longitudeDiff = maxLongitude - minLongitude;

        return [
            {
                // На десктопе нужен отступ слева для отображения сегментов
                lon: minLongitude - (deviceType.isMobile ? 0 : longitudeDiff),
                lat: maxLatitude,
            },
            {
                lon: maxLongitude,
                lat: minLatitude,
            },
        ];
    }, [airportsCoords, deviceType.isMobile]);

    return (
        <YandexMapsComponent
            className={className}
            hasZoomControl={false}
            defaultBounds={[topLeftCornerCoodrs, bottomRightCornerCoodrs]}
            minZoom={deviceType.isMobile ? 1 : 2}
            defaultMargins={[20, 20, 20, 20]}
            onLoadMap={setMapInstance}
        >
            {airportsCoords.map((airport, index) => {
                const nextAirport = airportsCoords[index + 1];

                if (!nextAirport) {
                    return null;
                }

                return (
                    <GeoObject
                        key={`line-${index}`}
                        geometry={{
                            type: 'Point',
                            coordinates: [
                                [airport.latitude, airport.longitude],
                                [nextAirport.latitude, nextAirport.longitude],
                            ],
                        }}
                        options={{
                            geodesic: true,
                            strokeWidth: 5,
                            strokeColor:
                                index === activeDepartureStationIndex
                                    ? EColors.BLACK
                                    : EColors.GRAY,
                        }}
                    />
                );
            })}

            {mapInstance &&
                airportsCoords.map((airport, index) => {
                    return (
                        <YandexMapPlacemark
                            key={`circle-${index}`}
                            options={{
                                iconLayout:
                                    mapInstance.templateLayoutFactory.createClass(
                                        renderToString(
                                            <div className={cx('station')} />,
                                        ),
                                    ),
                            }}
                            geometry={[airport.latitude, airport.longitude]}
                        />
                    );
                })}
        </YandexMapsComponent>
    );
};

export default React.memo(Map);
