import React, {useMemo} from 'react';

import {
    EAviaColor,
    IAviaFlightStatus,
    IAviaSegmentDuration,
    IAviaSegmentPoint,
} from 'server/api/AviaGatewayApi/types/IAviaGetFlightPageApiResponse';
import {IWithClassName} from 'types/withClassName';
import {ISegmentPlatformProps} from 'projects/avia/pages/FlightPage/components/Segments/components/Segment/types/ISegmentPlatformProps';

import {formatDate} from 'utilities/dateUtils';
import {TIME} from 'utilities/dateUtils/formats';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import getLabelStatusTheme from 'projects/avia/pages/FlightPage/components/Segments/components/Segment/utilities/getLabelStatusTheme';

import * as i18nFlightBlock from 'i18n/aviaFlight';

import Box from 'components/Box/Box';
import Text from 'components/Text/Text';
import Label from 'components/Label/Label';
import PointSettlementAndStation from 'projects/avia/pages/FlightPage/components/Segments/components/Segment/components/PointSettlementAndStation/PointSettlementAndStation';
import Separator from 'components/Separator/Separator';
import Terminal from 'projects/avia/pages/FlightPage/components/Segments/components/Segment/components/Terminal/Terminal';
import SegmentDesktop from 'projects/avia/pages/FlightPage/components/Segments/components/Segment/components/SegmentDesktop/SegmentDesktop';
import SegmentMobile from 'projects/avia/pages/FlightPage/components/Segments/components/Segment/components/SegmentMobile/SegmentMobile';
import Header from 'projects/avia/pages/FlightPage/components/Segments/components/Segment/components/Header/Header';
import CheckInDesks from 'projects/avia/pages/FlightPage/components/Segments/components/Segment/components/CheckInDesks/CheckInDesks';
import Gate from 'projects/avia/pages/FlightPage/components/Segments/components/Segment/components/Gate/Gate';

import cx from './Segment.scss';

interface ISegmentProps extends IWithClassName {
    isOpened: boolean;
    withHeader: boolean;
    status: IAviaFlightStatus;
    departure: IAviaSegmentPoint;
    arrival: IAviaSegmentPoint;
    duration: IAviaSegmentDuration;
    lastUpdate: string;
    mapBlock: React.ReactNode;
    disclaimerBlock: React.ReactNode;
    startDepartureFormattedDate: string;
}

const Segment: React.FC<ISegmentProps> = props => {
    const {
        className,
        isOpened,
        withHeader,
        status,
        departure,
        arrival,
        duration,
        lastUpdate,
        mapBlock,
        disclaimerBlock,
        startDepartureFormattedDate,
    } = props;

    const deviceType = useDeviceType();

    const departureActualAndScheduleTimeAreSame =
        departure.actualTime === departure.scheduledTime;
    const arrivalActualAndScheduleTimeAreSame =
        arrival.actualTime === arrival.scheduledTime;
    const currentDepartureFormattedTime = formatDate(
        departure.actualTime || departure.scheduledTime,
        TIME,
    );
    const currentArrivalFormattedTime = formatDate(
        arrival.actualTime || arrival.scheduledTime,
        TIME,
    );

    const isLateDeparture = departure.timeColor === EAviaColor.RED;
    const isLateArrival = arrival.timeColor === EAviaColor.RED;
    const isDiffDepartureDate =
        startDepartureFormattedDate !== departure.formattedDate;
    const isDiffArrivalDate =
        startDepartureFormattedDate !== arrival.formattedDate;

    const headerBlock = useMemo(() => {
        if (!withHeader) {
            return null;
        }

        return (
            <Header
                isOpened={isOpened}
                departure={departure}
                arrival={arrival}
                departureFormattedDate={currentDepartureFormattedTime}
                arrivalFormattedDate={currentArrivalFormattedTime}
                isDiffDepartureDate={isDiffDepartureDate}
                isDiffArrivalDate={isDiffArrivalDate}
            />
        );
    }, [
        arrival,
        currentArrivalFormattedTime,
        currentDepartureFormattedTime,
        departure,
        isDiffArrivalDate,
        isDiffDepartureDate,
        isOpened,
        withHeader,
    ]);

    const statusLabelBlock = useMemo(() => {
        const statusLabelTheme = getLabelStatusTheme(status.textColor);

        return <Label theme={statusLabelTheme}>{status.text}</Label>;
    }, [status.text, status.textColor]);

    const checkInDesksBlock = useMemo(() => {
        if (!status.checkInDesks) {
            return null;
        }

        return <CheckInDesks checkInDesks={status.checkInDesks} />;
    }, [status.checkInDesks]);

    const gateBlock = useMemo(() => {
        if (!departure.gate) {
            return null;
        }

        return <Gate gate={departure.gate} />;
    }, [departure.gate]);

    const baggageCarouselsBlock = useMemo(() => {
        if (!status.baggageCarousels) {
            return null;
        }

        return (
            <Text size="s" tag="div">
                {`${i18nFlightBlock.baggageCarousels()}: `}

                <Text size="s" weight="medium">
                    {status.baggageCarousels}
                </Text>
            </Text>
        );
    }, [status.baggageCarousels]);

    const departureSettlementAndStationBlock = useMemo(() => {
        return (
            <PointSettlementAndStation
                className={cx('departureSettlementAndStation')}
                settlement={departure.settlementTitle}
                stationTitle={departure.stationTitle}
                stationCode={departure.stationCode}
            />
        );
    }, [
        departure.settlementTitle,
        departure.stationCode,
        departure.stationTitle,
    ]);

    const arrivalSettlementAndStationBlock = useMemo(() => {
        return (
            <PointSettlementAndStation
                className={cx('arrivalSettlementAndStation')}
                settlement={arrival.settlementTitle}
                stationTitle={arrival.stationTitle}
                stationCode={arrival.stationCode}
            />
        );
    }, [arrival.settlementTitle, arrival.stationCode, arrival.stationTitle]);

    const departureTitleBlock = useMemo(() => {
        return (
            <Text
                className={cx('departureTitle', {
                    departureTitle_late: isLateDeparture,
                })}
                size="s"
                tag="div"
            >
                {i18nFlightBlock.departureTitle()}{' '}
                {deviceType.isDesktop && isDiffDepartureDate
                    ? departure.formattedDate
                    : ''}
            </Text>
        );
    }, [
        departure.formattedDate,
        deviceType.isDesktop,
        isDiffDepartureDate,
        isLateDeparture,
    ]);

    const departureTimeBlock = useMemo(() => {
        return (
            <Text
                className={cx('departureTime', {
                    departureTime_late: isLateDeparture,
                })}
                size="xxl"
            >
                <strong>{currentDepartureFormattedTime}</strong>

                {deviceType.isMobile && isDiffDepartureDate
                    ? ` ${departure.formattedDate}`
                    : ''}
            </Text>
        );
    }, [
        currentDepartureFormattedTime,
        departure.formattedDate,
        deviceType.isMobile,
        isDiffDepartureDate,
        isLateDeparture,
    ]);

    const timeSeparatorBlock = useMemo(() => {
        return <Separator className={cx('timeSeparator')} />;
    }, []);

    const arrivalTitleBlock = useMemo(() => {
        return (
            <Text
                className={cx('arrivalTitle', {
                    arrivalTitle_late: isLateArrival,
                })}
                size="s"
                tag="div"
            >
                {i18nFlightBlock.arrivalTitle()}{' '}
                {deviceType.isDesktop && isDiffArrivalDate
                    ? arrival.formattedDate
                    : ''}
            </Text>
        );
    }, [
        arrival.formattedDate,
        deviceType.isDesktop,
        isDiffArrivalDate,
        isLateArrival,
    ]);

    const arrivalTimeBlock = useMemo(() => {
        return (
            <Text
                className={cx('arrivalTime', {
                    arrivalTime_late: isLateArrival,
                })}
                size="xxl"
            >
                <strong>{currentArrivalFormattedTime}</strong>

                {deviceType.isMobile && isDiffArrivalDate
                    ? ` ${arrival.formattedDate}`
                    : ''}
            </Text>
        );
    }, [
        arrival.formattedDate,
        currentArrivalFormattedTime,
        deviceType.isMobile,
        isDiffArrivalDate,
        isLateArrival,
    ]);

    const departureScheduleTimeBlock = useMemo(() => {
        return (
            <Text
                className={cx('scheduleTime', {
                    scheduleTime_hidden:
                        !departure.actualTime ||
                        departureActualAndScheduleTimeAreSame,
                })}
                size="s"
                color="secondary"
            >
                {deviceType.isMobile ? `${i18nFlightBlock.bySchedule()} ` : ''}
                {formatDate(departure.scheduledTime, TIME)}
            </Text>
        );
    }, [
        departure.actualTime,
        departure.scheduledTime,
        departureActualAndScheduleTimeAreSame,
        deviceType.isMobile,
    ]);

    const arrivalScheduleTimeBlock = useMemo(() => {
        return (
            <Text
                className={cx('scheduleTime', {
                    scheduleTime_hidden:
                        !arrival.actualTime ||
                        arrivalActualAndScheduleTimeAreSame,
                })}
                size="s"
                color="secondary"
            >
                {deviceType.isMobile ? `${i18nFlightBlock.bySchedule()} ` : ''}
                {formatDate(arrival.scheduledTime, TIME)}
            </Text>
        );
    }, [
        arrival.actualTime,
        arrival.scheduledTime,
        arrivalActualAndScheduleTimeAreSame,
        deviceType.isMobile,
    ]);

    const durationBlock = useMemo(() => {
        return (
            <Text size="s" color="secondary">
                {duration.total}
            </Text>
        );
    }, [duration.total]);

    const departureTerminalBlock = useMemo(() => {
        return <Terminal terminal={departure.terminal} />;
    }, [departure.terminal]);

    const arrivalTerminalBlock = useMemo(() => {
        return (
            <Terminal
                className={cx('arrivalTerminal')}
                terminal={arrival.terminal}
            />
        );
    }, [arrival.terminal]);

    const lastUpdateBlock = useMemo(() => {
        return (
            <Box textSize="s" above={3} textColor="secondary">
                {lastUpdate}
            </Box>
        );
    }, [lastUpdate]);

    const commonProps: ISegmentPlatformProps = {
        isOpened,
        headerBlock,
        statusLabelBlock,
        checkInDesksBlock,
        gateBlock,
        baggageCarouselsBlock,
        departureSettlementAndStationBlock,
        departureTitleBlock,
        departureTimeBlock,
        departureTerminalBlock,
        departureScheduleTimeBlock,
        arrivalSettlementAndStationBlock,
        arrivalTitleBlock,
        arrivalTimeBlock,
        timeSeparatorBlock,
        arrivalScheduleTimeBlock,
        arrivalTerminalBlock,
        lastUpdateBlock,
    };

    if (deviceType.isDesktop) {
        return (
            <SegmentDesktop
                {...commonProps}
                className={cx('root', 'root_desktop', className)}
                durationBlock={durationBlock}
            />
        );
    }

    return (
        <SegmentMobile
            {...commonProps}
            className={cx('root', 'root_mobile', className)}
            mapBlock={mapBlock}
            disclaimerBlock={disclaimerBlock}
        />
    );
};

export default React.memo(Segment);
