import React, {useMemo} from 'react';
import {useLocation} from 'react-router-dom';

import {IAviaSegmentPoint} from 'server/api/AviaGatewayApi/types/IAviaGetFlightPageApiResponse';

import getUrlWithUpdatedQueryByLocation from 'utilities/url/getUrlWithUpdatedQueryByLocation';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import getHeaderTitle from 'projects/avia/pages/FlightPage/components/Segments/components/Segment/components/Header/utilities/getHeaderTitle';
import useSaveElementPositionCallback from 'utilities/hooks/useSaveElementPositionCallback';

import * as i18nCommonBlock from 'i18n/common';

import Text from 'components/Text/Text';
import Link from 'components/Link/Link';
import RotatingArrowIcon from 'components/RotatingArrowIcon/RotatingArrowIcon';

import cx from './Header.scss';

interface IHeaderProps {
    isOpened: boolean;
    departure: IAviaSegmentPoint;
    arrival: IAviaSegmentPoint;
    departureFormattedDate: string;
    arrivalFormattedDate: string;
    isDiffDepartureDate: boolean;
    isDiffArrivalDate: boolean;
}

const Header: React.FC<IHeaderProps> = props => {
    const {
        isOpened,
        departure,
        arrival,
        departureFormattedDate,
        arrivalFormattedDate,
        isDiffDepartureDate,
        isDiffArrivalDate,
    } = props;

    const deviceType = useDeviceType();
    const location = useLocation();

    const handleClick = useSaveElementPositionCallback();

    const contentBlock = useMemo(() => {
        const directionText = i18nCommonBlock.direction({
            from: departure.settlementTitle,
            to: arrival.settlementTitle,
        });

        const directionBlock = deviceType.isDesktop ? (
            <strong>{directionText}</strong>
        ) : (
            directionText
        );

        if (isOpened) {
            return (
                <Text size={deviceType.isMobile ? 'm' : 's'} tag="div">
                    {directionBlock}
                </Text>
            );
        }

        return (
            <Link
                className={cx('root', deviceMods('root', deviceType))}
                to={getUrlWithUpdatedQueryByLocation(
                    {from: isOpened ? undefined : departure.stationCode},
                    location,
                )}
            >
                {directionBlock}

                {deviceType.isDesktop && (
                    <span className={cx('time')}>
                        {getHeaderTitle(
                            departureFormattedDate,
                            departure.formattedDate,
                            isDiffDepartureDate,
                            arrivalFormattedDate,
                            arrival.formattedDate,
                            isDiffArrivalDate,
                        )}
                    </span>
                )}

                <RotatingArrowIcon
                    className={cx('arrowIcon')}
                    rotated={isOpened}
                />
            </Link>
        );
    }, [
        arrival.formattedDate,
        arrival.settlementTitle,
        arrivalFormattedDate,
        departure.formattedDate,
        departure.settlementTitle,
        departure.stationCode,
        departureFormattedDate,
        deviceType,
        isDiffArrivalDate,
        isDiffDepartureDate,
        isOpened,
        location,
    ]);

    return (
        <div
            onClick={isOpened || deviceType.isDesktop ? undefined : handleClick}
        >
            {contentBlock}
        </div>
    );
};

export default React.memo(Header);
