import {FC, memo, useCallback, useEffect, useMemo} from 'react';
import {useDispatch} from 'react-redux';

import ERefundPartState from 'server/api/GenericOrderApi/types/common/refund/ERefundPartState';
import {IRefundPartInfo} from 'server/api/GenericOrderApi/types/common/refund/IRefundPartInfo';
import {IWithClassName} from 'types/withClassName';
import {IGenericOrderInfo} from 'server/api/GenericOrderApi/types/common/IGenericOrderInfo';
import {EBusesGoal} from 'utilities/metrika/types/goals/buses';

import initRefundAction from 'reducers/account/genericOrder/refund/thunk/initRefundAction';
import {updateHappyPageGenericOrderAction} from 'reducers/happyPage/actions';
import {getGenericOrderItemActions} from 'reducers/account/genericOrder/item/actions';

import {getOrderUrl} from 'projects/account/utilities/urls/getOrderUrl';
import {reachGoal} from 'utilities/metrika';

import * as i18nBlock from 'i18n/account-OrderTrains-Actions';

import CommonOrderActions, {
    EOrderActionsSource,
    IOrderActionsCancel,
    IOrderActionsDetails,
} from 'components/OrderActions/OrderActions';
import useDownloadAndPrint from 'projects/buses/components/OrderActions/hooks/useDownloandAndPrint';
import usePollDownloadBlankToken from 'projects/buses/components/OrderActions/hooks/usePollDownloadBlankToken';

interface IOrderActionsProps extends IWithClassName {
    source: EOrderActionsSource;
    orderId: string;
    downloadBlankToken: string | undefined;
    ticketsWithPossibleRefundCount?: number;
    refundPartInfo?: IRefundPartInfo | null;
    isRefunding?: boolean;
}

const OrderActions: FC<IOrderActionsProps> = props => {
    const {
        className,
        source,
        orderId,
        downloadBlankToken,
        ticketsWithPossibleRefundCount,
        refundPartInfo,
        isRefunding,
    } = props;

    const dispatch = useDispatch();

    const orderDetails = useMemo((): IOrderActionsDetails | undefined => {
        if (source === EOrderActionsSource.ORDER_PAGE) {
            return undefined;
        }

        return {
            url: getOrderUrl(orderId),
        };
    }, [orderId, source]);

    const {print, download} = useDownloadAndPrint(orderId, downloadBlankToken);

    const cancel = useMemo((): IOrderActionsCancel | undefined => {
        if (
            source === EOrderActionsSource.HAPPY_PAGE ||
            !refundPartInfo ||
            refundPartInfo.state !== ERefundPartState.ENABLED
        ) {
            return undefined;
        }

        return {
            title:
                ticketsWithPossibleRefundCount === 1
                    ? i18nBlock.refundTicket()
                    : i18nBlock.refundAllTickets(),
            isDisabled: isRefunding,
            onClick(): void {
                dispatch(initRefundAction([refundPartInfo.context]));
            },
        };
    }, [
        dispatch,
        isRefunding,
        refundPartInfo,
        source,
        ticketsWithPossibleRefundCount,
    ]);

    const handleTokenReady = useCallback(
        ({order}: {order: IGenericOrderInfo}) => {
            if (source === EOrderActionsSource.HAPPY_PAGE) {
                dispatch(updateHappyPageGenericOrderAction(order));
            } else {
                dispatch(getGenericOrderItemActions.success(order));
            }
        },
        [dispatch, source],
    );

    usePollDownloadBlankToken({
        orderId,
        initialDownloadBlankToken: downloadBlankToken,
        onTokenReady: handleTokenReady,
    });

    useEffect(() => {
        if (downloadBlankToken) {
            reachGoal(EBusesGoal.BLANK_GENERATED);
        }
    }, [downloadBlankToken]);

    return (
        <CommonOrderActions
            className={className}
            source={source}
            orderDetails={orderDetails}
            print={print}
            download={download}
            cancel={cancel}
        />
    );
};

export default memo(OrderActions);
