import {useCallback, useMemo} from 'react';

import getGenericDownloadBlankUrl from 'projects/account/pages/Order/utilities/downloadBlank/getGenericDownloadBlankUrl';
import getDownloadFilename from 'projects/account/utilities/downloadBlank/getDownloadFilename';

import {
    IOrderActionsDownload,
    IOrderActionsPrint,
} from 'components/OrderActions/OrderActions';
import Spinner from 'components/Spinner/Spinner';

export default function useDownloadAndPrint(
    orderId: string,
    downloadBlankToken: string | undefined,
): {
    print: IOrderActionsPrint | undefined;
    download: IOrderActionsDownload | undefined;
} {
    const downloadBlankUrl = useMemo(() => {
        if (!downloadBlankToken) {
            return undefined;
        }

        return getGenericDownloadBlankUrl({
            orderId,
            token: downloadBlankToken,
        });
    }, [downloadBlankToken, orderId]);

    const renderSpinnerIcon = useCallback(
        (className?: string): JSX.Element => (
            <Spinner className={className} size="xxs" />
        ),
        [],
    );

    return {
        print: useMemo(
            (): IOrderActionsPrint | undefined => ({
                url: downloadBlankUrl || '',
                isDisabled: !downloadBlankUrl,
                renderIcon: downloadBlankUrl ? undefined : renderSpinnerIcon,
            }),
            [downloadBlankUrl, renderSpinnerIcon],
        ),

        download: useMemo(
            (): IOrderActionsDownload | undefined => ({
                url: downloadBlankUrl || '',
                fileName: getDownloadFilename('order', orderId, 'pdf'),
                isDisabled: !downloadBlankUrl,
                renderIcon: downloadBlankUrl ? undefined : renderSpinnerIcon,
            }),
            [downloadBlankUrl, orderId, renderSpinnerIcon],
        ),
    };
}
