import {useEffect} from 'react';
import axios from 'axios';
import {useDispatch} from 'react-redux';

import {SECOND} from 'utilities/dateUtils/constants';

import {IGenericOrderInfo} from 'server/api/GenericOrderApi/types/common/IGenericOrderInfo';

import {delay} from 'utilities/async/delay';
import isBusService from 'projects/buses/utilities/booking/isBusesService';
import {logError} from 'utilities/logger/logError';

import {genericOrderBrowserProvider} from 'serviceProvider/genericOrder/genericOrderBrowserProvider';

export default function usePollDownloadBlankToken({
    initialDownloadBlankToken,
    orderId,
    onTokenReady,
}: {
    initialDownloadBlankToken: string | undefined;
    orderId: string;
    onTokenReady(data: {
        order: IGenericOrderInfo;
        downloadBlankToken: string;
    }): void;
}): void {
    const dispatch = useDispatch();

    useEffect(() => {
        if (initialDownloadBlankToken) {
            return;
        }

        const abortController = new AbortController();

        let previousHash = null;

        (async (): Promise<void> => {
            try {
                // eslint-disable-next-line no-constant-condition
                while (true) {
                    await delay(2 * SECOND);

                    const {versionHash} =
                        await genericOrderBrowserProvider.getOrderState(
                            {
                                orderId,
                            },
                            {signal: abortController.signal},
                        );

                    if (versionHash === previousHash) {
                        continue;
                    }

                    previousHash = versionHash;

                    const order = await genericOrderBrowserProvider.getOrder(
                        {
                            orderId,
                        },
                        {signal: abortController.signal},
                    );

                    const service = order.services.find(isBusService);

                    if (!service) {
                        break;
                    }

                    const {
                        busInfo: {downloadBlankToken: newDownloadBlankToken},
                    } = service;

                    if (!newDownloadBlankToken) {
                        continue;
                    }

                    onTokenReady({
                        order,
                        downloadBlankToken: newDownloadBlankToken,
                    });

                    break;
                }
            } catch (err) {
                if (!axios.isCancel(err)) {
                    logError(
                        {
                            message: '[YATRAVEL][BUSES] Загрузка билета',
                        },
                        err,
                    );
                }
            }
        })();

        return (): void => {
            abortController.abort();
        };
    }, [dispatch, initialDownloadBlankToken, orderId, onTokenReady]);
}
