import React, {memo} from 'react';
import moment from 'moment-timezone';

import {IBusesRide} from 'types/buses/booking/IBusesRide';

import {formatDate} from 'utilities/dateUtils';
import {HUMAN_DATE_WITH_FULL_WEEKDAY} from 'utilities/dateUtils/formats';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import getPointTitle from 'projects/buses/utilities/getPointTitle';

import * as i18nCommon from 'i18n/common';

import Box from 'components/Box/Box';
import Segment from 'projects/buses/components/Segment/Segment';
import Flex from 'components/Flex/Flex';
import Separator from 'components/Separator/Separator';
import Text from 'components/Text/Text';
import useCommonSegmentProps from 'projects/buses/components/OrderSegment/hooks/useCommonSegmentProps/useCommonSegmentProps';
import useStationsMap from 'projects/buses/components/StationMap/hooks/useStationsMap';
import StationMap from 'projects/buses/components/StationMap/StationMap';

import cx from './OrderSegment.scss';

interface IOrderSegmentProps {
    ride: IBusesRide;
}

const OrderSegment: React.FC<IOrderSegmentProps> = props => {
    const {
        ride: {
            departure,
            arrival,
            titlePointFrom,
            pointFrom,
            titlePointTo,
            pointTo,
            duration,
            carrier,
            busDescription,
            benefits,
            refundConditions,
        },
    } = props;

    const deviceType = useDeviceType();

    const {
        points: mapPoints,
        setPoints: setMapPoints,
        closeMap,
    } = useStationsMap();

    const commonSegmentProps = useCommonSegmentProps({
        pointFrom,
        pointTo,
        departure,
        arrival,
        duration,
        carrier,
        busDescription,
        facilities: benefits,
        refundConditions,
        setMapPoints,
        cx,
    });

    return (
        <Box
            className={cx('root', deviceMods('root', deviceType))}
            between={deviceType.isMobile ? 0 : 5}
        >
            <Flex
                flexDirection={deviceType.isMobile ? 'column' : 'row'}
                between={deviceType.isMobile ? 1 : 2}
                inline={deviceType.isDesktop}
            >
                <Text
                    size={deviceType.isMobile ? 'l' : 'xl'}
                    weight={deviceType.isMobile ? 'medium' : 'bold'}
                >
                    {i18nCommon.direction({
                        from: getPointTitle(titlePointFrom),
                        to: getPointTitle(titlePointTo),
                    })}
                </Text>

                <Text size={deviceType.isMobile ? 'm' : 'xl'}>
                    {formatDate(
                        moment.utc(departure),
                        HUMAN_DATE_WITH_FULL_WEEKDAY,
                    )}
                </Text>
            </Flex>

            {deviceType.isMobile && <Separator margin={3} />}

            <Segment {...commonSegmentProps} />

            <StationMap points={mapPoints} onClose={closeMap} />
        </Box>
    );
};

export default memo(OrderSegment);
