import React, {useEffect, useMemo, useRef} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import isEmpty from 'lodash/isEmpty';

import {BUSES_SEARCH_FORM_NAME} from 'constants/reduxForm';
import {BUSES_SPECIAL_WHEN_LIST} from 'projects/buses/constants/when';

import {IWithClassName} from 'types/withClassName';
import {ESearchFormFieldName} from 'components/SearchForm/types';
import {
    IBusesSearchFormErrors,
    IBusesSearchFormPointField,
    TBusesSearchFormDateField,
} from 'projects/buses/components/SearchForm/types';
import {EIndexGoal} from 'utilities/metrika/types/goals';
import {EBusesGoal} from 'utilities/metrika/types/goals/buses';
import {WHEN_SPECIAL_VALUE} from 'types/common/When';
import {IPreviousSearchFormSetFieldValuePayload} from 'reducers/common/previousSearchForm/types';

import requestCalendarThunkAction from 'reducers/buses/calendar/thunk/requestCalendarThunkAction';
import {
    setBusesSearchFormFromFieldAction,
    setBusesSearchFormStartDateFieldAction,
    setBusesSearchFormToFieldAction,
} from 'reducers/buses/searchForm/actions';
import requestSearchSuggestsThunkAction from 'reducers/buses/searchSuggests/thunk/requestSearchSuggestsThunkAction';
import {previousSearchFormSetFieldValue} from 'reducers/common/previousSearchForm/actions';

import searchFormSelector from 'selectors/buses/searchForm/searchFormSelector';

import validateForm from 'projects/buses/components/SearchForm/utilities/validateForm';
import useImmutableCallback from 'utilities/hooks/useImmutableCallback';
import {reachGoal} from 'utilities/metrika';
import getSearchUrl from 'projects/buses/utilities/urls/getSearchUrl';
import history from 'utilities/browserHistory/browserHistory';
import getCalendarPrices from 'projects/buses/components/SearchForm/utilities/getCalendarPrices';
import getSuggestTitleAndDescription from 'projects/buses/components/SearchForm/utilities/getSuggestTitleAndDescription';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {isTransportCrossSearch} from 'utilities/searchForm/isCrossSearch';

import * as i18nBlock from 'i18n/components';

import CommonSearchForm, {
    IBaseSearchFormProps,
    IOnDateClickParams,
} from '../../../../components/SearchForm/SearchForm';
import useRequestSuggestsWithThrottle from 'components/SearchForm/hooks/useRequestSuggestsWithThrottle';
import useSuggestInputFocusHandler from 'components/SearchForm/hooks/useSuggestInputFocusHandler';
import useCalendarConversionExpMetrics from 'components/DatePicker/hooks/useCalendarConversionExpMetrics';
import {ESuggestSource} from 'components/SearchSuggest/SearchSuggest';

const FIELD_NAMES = [
    ESearchFormFieldName.FROM,
    ESearchFormFieldName.TO,
    ESearchFormFieldName.START_DATE,
];

interface ISearchFormProps extends IBaseSearchFormProps, IWithClassName {
    needToSetFromSuggestByGeoPosition?: boolean;
    onSubmit?({searchLink}: {searchLink: string}): void;
}

const SearchForm: React.FC<ISearchFormProps> = props => {
    const {
        needToSetFromSuggestByGeoPosition = false,
        onSubmit,
        ...restProps
    } = props;

    const {
        fromField,
        toField,
        startDateField,
        searchSuggests,
        previousFormValues,
        calendar,
    } = useSelector(searchFormSelector);
    const {reachSearchInteraction, reachCalendarInteraction} =
        useCalendarConversionExpMetrics(EBusesGoal);

    const dispatch = useDispatch();

    const isInitialSuggestsRequestRef = useRef(true);

    const setFromPoint = useImmutableCallback(
        (point: IBusesSearchFormPointField) => {
            dispatch(setBusesSearchFormFromFieldAction(point));
        },
    );

    const setToPoint = useImmutableCallback(
        (point: IBusesSearchFormPointField) => {
            dispatch(setBusesSearchFormToFieldAction(point));
        },
    );

    const setStartDate = useImmutableCallback(
        (date: TBusesSearchFormDateField) => {
            dispatch(setBusesSearchFormStartDateFieldAction(date));
        },
    );

    const setPreviousSearchFormValue = useImmutableCallback(
        (data: IPreviousSearchFormSetFieldValuePayload) => {
            dispatch(previousSearchFormSetFieldValue(data));
        },
    );

    const requestCalendar = useImmutableCallback(() => {
        if (
            typeof fromField.selectedValue === 'boolean' ||
            typeof toField.selectedValue === 'boolean'
        ) {
            return;
        }

        dispatch(
            requestCalendarThunkAction({
                fromPointKey: fromField.selectedValue.pointKey,
                toPointKey: toField.selectedValue.pointKey,
            }),
        );
    });

    const requestSuggests = useImmutableCallback(
        (from: IBusesSearchFormPointField, to: IBusesSearchFormPointField) => {
            // Не фетчим саджесты во время загрузки cross-search
            if (
                from.source === ESuggestSource.CROSS_SEARCH &&
                to.source === ESuggestSource.CROSS_SEARCH &&
                !from.selectedValue &&
                !to.selectedValue
            ) {
                return;
            }

            dispatch(
                requestSearchSuggestsThunkAction({
                    fromField: from,
                    toField: to,
                    needToSetByGeoPointIfPossible:
                        needToSetFromSuggestByGeoPosition &&
                        isInitialSuggestsRequestRef.current,
                }),
            );

            isInitialSuggestsRequestRef.current = false;
        },
    );

    const requestSuggestsWithThrottle =
        useRequestSuggestsWithThrottle(requestSuggests);

    const handleSuggestInputFocus = useSuggestInputFocusHandler(
        fromField,
        toField,
        requestSuggestsWithThrottle,
    );

    const handleReverseClick = useImmutableCallback(() => {
        reachGoal(EBusesGoal.POINT_POINT_SWAP_BUTTON_CLICK);
    });

    const handleCalendarShow = useImmutableCallback(() => {
        reachGoal(EBusesGoal.CALENDAR_OPEN);

        if (fromField.selectedValue && toField.selectedValue) {
            reachCalendarInteraction();
        }
    });

    const handleDateClick = useImmutableCallback(
        (params?: IOnDateClickParams) => {
            if (!params) {
                return;
            }

            const {startDate} = params;

            if (!startDate) {
                return;
            }

            reachGoal(EBusesGoal.CALENDAR_DATE_SELECTED);
        },
    );

    const handleSubmit = useImmutableCallback(
        ({
            handleTrySubmit,
            formErrors,
            setHasSubmitFailed,
        }: {
            handleTrySubmit(): void;
            formErrors: IBusesSearchFormErrors;
            setHasSubmitFailed(val: boolean): void;
        }) => {
            reachGoal(EIndexGoal.BUSES_SEARCH_BUTTON);
            reachSearchInteraction();

            handleTrySubmit();

            if (isEmpty(formErrors)) {
                const fromSelectedValue = fromField.selectedValue;
                const toSelectedValue = toField.selectedValue;

                if (
                    typeof fromSelectedValue !== 'boolean' &&
                    typeof toSelectedValue !== 'boolean'
                ) {
                    const when =
                        startDateField === null
                            ? WHEN_SPECIAL_VALUE.TODAY
                            : startDateField;

                    const searchLink = getSearchUrl({
                        fromSlug: fromSelectedValue.slug,
                        toSlug: toSelectedValue.slug,
                        lastSearchTimeMarker: String(Date.now()),
                        when,
                    });

                    history?.push(searchLink);

                    onSubmit?.({searchLink});

                    if (startDateField === null) {
                        dispatch(
                            setBusesSearchFormStartDateFieldAction(
                                WHEN_SPECIAL_VALUE.TODAY,
                            ),
                        );
                    }

                    if (isTransportCrossSearch(fromField, toField)) {
                        reachGoal(EBusesGoal.CROSS_SEARCH_SUBMIT);
                    }
                }
            } else {
                setHasSubmitFailed(true);
            }
        },
    );

    const handleFastDateClick = useImmutableCallback((value: string) => {
        switch (value) {
            case WHEN_SPECIAL_VALUE.TODAY:
                reachGoal(EBusesGoal.CALENDAR_FASTDATE_TODAY);

                break;
            case WHEN_SPECIAL_VALUE.TOMORROW:
                reachGoal(EBusesGoal.CALENDAR_FASTDATE_TOMORROW);

                break;
        }
    });

    const calendarPrices = useMemo(() => {
        return getCalendarPrices(
            calendar,
            fromField.selectedValue,
            toField.selectedValue,
        );
    }, [calendar, fromField.selectedValue, toField.selectedValue]);

    useEffect(() => {
        requestSuggestsWithThrottle(fromField, toField);
    }, [fromField, requestSuggestsWithThrottle, toField]);

    return (
        <CommonSearchForm
            searchFormName={BUSES_SEARCH_FORM_NAME}
            fieldsNames={FIELD_NAMES}
            fromField={fromField}
            toField={toField}
            startDateField={startDateField}
            fromSearchSuggests={searchSuggests[ESearchFormFieldName.FROM].value}
            toSearchSuggests={searchSuggests[ESearchFormFieldName.TO].value}
            previousFormValues={previousFormValues}
            calendarPrices={calendarPrices}
            validateForm={validateForm}
            uniqueSuggestValueName="objId"
            getSuggestTitleAndDescription={getSuggestTitleAndDescription}
            fromPointPlaceholder={i18nBlock.busesSearchFormDotFromFieldPlaceholder()}
            toPointPlaceholder={i18nBlock.busesSearchFormDotToFieldPlaceholder()}
            startDatePlaceholder={i18nBlock.busesSearchFormDotStartDatePlaceholder()}
            specialWhenButtons={BUSES_SPECIAL_WHEN_LIST}
            onFastDateClick={handleFastDateClick}
            setFromPoint={setFromPoint}
            setToPoint={setToPoint}
            setStartDate={setStartDate}
            setPreviousSearchFormValue={setPreviousSearchFormValue}
            requestCalendar={requestCalendar}
            onReverseClick={handleReverseClick}
            onSuggestInputFocus={handleSuggestInputFocus}
            onShowCalendar={handleCalendarShow}
            onDateClick={handleDateClick}
            onSubmit={handleSubmit}
            {...prepareQaAttributes('buses-search-form')}
            {...restProps}
        />
    );
};

export default React.memo(SearchForm);
