import {IBusesSuggest} from 'types/buses/common/IBusesSuggest';
import {
    IBusesSearchFormErrors,
    IBusesSearchFormPointField,
    IBusesSearchFormValues,
} from 'projects/buses/components/SearchForm/types';

import * as i18nBlock from 'i18n/components';

const isEmptyDirectionValue = ({
    inputValue,
}: IBusesSearchFormPointField): boolean => inputValue === '';

const isEmptySelectedValue = ({
    selectedValue,
}: IBusesSearchFormPointField): boolean => selectedValue === false;

const isSelectedValue = (
    selectedValue: false | IBusesSuggest,
): selectedValue is IBusesSuggest => typeof selectedValue !== 'boolean';

const isDirectionNotFound = ({
    inputValue,
    selectedValue,
}: IBusesSearchFormPointField): boolean =>
    inputValue !== '' && selectedValue === false;

const isEqualDirections = (
    from: IBusesSearchFormPointField,
    to: IBusesSearchFormPointField,
): boolean => {
    const fromSelectedValue = from.selectedValue;
    const toSelectedValue = to.selectedValue;

    return (
        isSelectedValue(fromSelectedValue) &&
        isSelectedValue(toSelectedValue) &&
        fromSelectedValue.objId === toSelectedValue.objId
    );
};

export default function validateForm(
    fieldsValues: IBusesSearchFormValues,
): IBusesSearchFormErrors {
    if (fieldsValues) {
        const {from, to} = fieldsValues;
        const errors: {from?: string[]; to?: string[]} = {};

        if (isEmptySelectedValue(from)) {
            errors.from = [
                i18nBlock.busesSearchFormDotValidateDotNeedSelectFromField(),
            ];
        }

        if (isEmptySelectedValue(to)) {
            errors.to = [
                i18nBlock.busesSearchFormDotValidateDotNeedSelectToField(),
            ];
        }

        if (isEmptyDirectionValue(from)) {
            errors.from = [
                i18nBlock.busesSearchFormDotValidateDotFromFieldRequired(),
            ];
        }

        if (isEmptyDirectionValue(to)) {
            errors.to = [
                i18nBlock.busesSearchFormDotValidateDotToFieldRequired(),
            ];
        }

        if (isDirectionNotFound(from)) {
            errors.from = [
                i18nBlock.busesSearchFormDotValidateDotUnknownFromDirection(),
            ];
        }

        if (isDirectionNotFound(to)) {
            errors.to = [
                i18nBlock.busesSearchFormDotValidateDotUnknownToDirection(),
            ];
        }

        if (isEqualDirections(from, to)) {
            errors.to = [
                i18nBlock.busesSearchFormDotValidateDotEqualDirections(),
            ];
        }

        return errors;
    }

    return {};
}
