import React from 'react';

import {IWithClassName} from 'types/withClassName';
import {EBusBenefit} from 'types/buses/booking/IBusesRide';

import {getDuration} from 'projects/avia/lib/date/getDuration';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import * as i18nBlock from 'i18n/avia-Duration';

import Text from 'components/Text/Text';
import Flex from 'components/Flex/Flex';
import Intersperse from 'components/Intersperse/Intersperse';
import DotSeparator from 'components/DotSeparator/DotSeparator';
import Facilities from 'projects/buses/components/Segment/components/SegmentDesktop/components/Facilities/Facilities';

import cx from './AdditionalInfo.scss';

interface IAdditionalInfoProps extends IWithClassName, IWithQaAttributes {
    duration?: number | null;
    carrier?: string;
    busDescription?: string;
    facilities?: EBusBenefit[];
    isSecondaryCarrierText?: boolean;
}

const AdditionalInfo: React.FC<IAdditionalInfoProps> = props => {
    const {
        className,
        duration,
        carrier,
        busDescription,
        facilities,
        isSecondaryCarrierText,
    } = props;

    const deviceType = useDeviceType();

    if (!duration && !busDescription && !carrier) {
        return null;
    }

    return (
        <Flex className={className} between={1} flexDirection="column">
            {duration ? (
                <Text
                    className={cx('duration')}
                    size="m"
                    weight="medium"
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'duration',
                    })}
                >
                    {i18nBlock
                        .dDashHDashMDashFormat(getDuration(duration))
                        .trim()}
                </Text>
            ) : null}

            <Text size={deviceType.isMobile ? 's' : 'm'}>
                <Intersperse
                    separator={
                        <DotSeparator
                            className={cx('dotSeparator', {
                                dotSeparator_secondary: isSecondaryCarrierText,
                            })}
                        />
                    }
                >
                    {busDescription && (
                        <span
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'bus',
                            })}
                        >
                            {busDescription}
                        </span>
                    )}

                    {carrier && (
                        <Text
                            color={
                                isSecondaryCarrierText ? 'secondary' : 'primary'
                            }
                            size="inherit"
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'carrier',
                            })}
                        >
                            {carrier}
                        </Text>
                    )}
                </Intersperse>
            </Text>

            {facilities?.length ? <Facilities items={facilities} /> : null}
        </Flex>
    );
};

export default React.memo(AdditionalInfo);
