import React from 'react';
import {Moment} from 'moment-timezone';

import {IWithClassName} from 'types/withClassName';

import {HUMAN_SHORT, TIME} from 'utilities/dateUtils/formats';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {deviceModDesktop} from 'utilities/stylesUtils';

import * as i18nSegmentsBlock from 'i18n/buses-Segments';

import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';

import cx from './DateAndTime.scss';

interface IDateAndTimeProps extends IWithClassName, IWithQaAttributes {
    time: Moment | null;
}

const DateAndTime: React.FC<IDateAndTimeProps> = props => {
    const {className, time} = props;

    const deviceType = useDeviceType();

    const {isMobile, isDesktop} = deviceType;

    return (
        <Flex
            className={cx(deviceModDesktop('root', deviceType), className)}
            between={isMobile ? 0 : 2}
            inline={isDesktop}
            flexDirection={isMobile ? 'column-reverse' : 'row'}
        >
            {time ? (
                <>
                    <Text
                        size={isMobile ? 'l' : 'xl'}
                        weight="bold"
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'time',
                        })}
                    >
                        {time.format(TIME)}
                    </Text>

                    <Text
                        size={isMobile ? 'xs' : 's'}
                        color="secondary"
                        {...prepareQaAttributes({
                            parent: props,
                            current: 'date',
                        })}
                    >
                        {time.format(HUMAN_SHORT)}
                    </Text>
                </>
            ) : (
                <Text
                    className={cx('unknownArrival')}
                    size={isMobile ? 's' : 'm'}
                    color="secondary"
                    tag="div"
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'unknownArrival',
                    })}
                >
                    {i18nSegmentsBlock.unknownArrival()}
                </Text>
            )}
        </Flex>
    );
};

export default React.memo(DateAndTime);
