import {memo, FC, useMemo} from 'react';

import {IWithClassName} from 'types/withClassName';
import {IBusesCommonSegmentInfo} from 'projects/buses/components/Segment/types';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {CHAR_EM_DASH} from 'utilities/strings/charCodes';

import * as i18n from 'i18n/buses-Segments';
import * as i18nCommonBlock from 'i18n/common';

import Flex from 'components/Flex/Flex';
import DateAndTime from 'projects/buses/components/Segment/components/DateAndTime/DateAndTime';
import Duration from 'projects/buses/components/Segment/components/SegmentMobile/components/Duration/Duration';
import Box from 'components/Box/Box';
import SearchSegmentBookInfo from 'components/SearchSegmentBookInfo/SearchSegmentBookInfo';
import AdditionalInfo from 'projects/buses/components/Segment/components/AdditionalInfo/AdditionalInfo';
import LinkButton from 'components/LinkButton/LinkButton';
import RefundConditions from 'projects/buses/components/Segment/components/SegmentMobile/components/RefundConditions/RefundConditions';

import cx from './SegmentMobile.scss';

interface ISegmentMobileProps
    extends IBusesCommonSegmentInfo,
        IWithClassName,
        IWithQaAttributes {}

const SegmentMobile: FC<ISegmentMobileProps> = props => {
    const {
        className,
        departure,
        arrival,
        additionalInfo,
        book,
        refundConditions,
        onShowStationsOnMap,
    } = props;

    const stationsOnMapBlock = useMemo(() => {
        if (!onShowStationsOnMap) {
            return null;
        }

        return (
            <Box textSize="s" above={3}>
                <LinkButton onClick={onShowStationsOnMap}>
                    {i18n.stationMap()}
                </LinkButton>
            </Box>
        );
    }, [onShowStationsOnMap]);

    const refundConditionsBlock = useMemo(() => {
        if (!refundConditions) {
            return null;
        }

        return <RefundConditions text={refundConditions} />;
    }, [refundConditions]);

    return (
        <Flex className={cx('root', className)} flexDirection="column">
            <Flex alignItems="flex-end" inline>
                <DateAndTime
                    time={departure.time}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'departure',
                    })}
                />

                <Duration
                    className={cx('duration')}
                    duration={additionalInfo.duration}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'duration',
                    })}
                />

                <DateAndTime
                    className={cx('arrival')}
                    time={arrival.time}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'arrival',
                    })}
                />
            </Flex>

            <Box textSize="s" above={1}>
                <span
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'departure-station',
                    })}
                >
                    {departure.name}
                </span>

                {` ${CHAR_EM_DASH} `}

                <span
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'arrival-station',
                    })}
                >
                    {arrival.name}
                </span>
            </Box>

            <AdditionalInfo
                className={cx('additionalInfo')}
                carrier={additionalInfo.carrier}
                busDescription={additionalInfo.busDescription}
                isSecondaryCarrierText={
                    additionalInfo.isSecondaryBusAndCarrierText
                }
                {...prepareQaAttributes(props)}
            />

            {(stationsOnMapBlock || refundConditionsBlock) && (
                <Box inline nowrap between={5}>
                    {stationsOnMapBlock}
                    {refundConditionsBlock}
                </Box>
            )}

            {book && (
                <SearchSegmentBookInfo
                    className={cx('book')}
                    price={book.price}
                    info={book.freeSeatsInfo}
                    bookUrl={book.url}
                    onClick={book.onClick}
                    buttonText={i18nCommonBlock.choose()}
                    {...prepareQaAttributes(props)}
                />
            )}
        </Flex>
    );
};

export default memo(SegmentMobile);
