import {FC, memo, useEffect, useMemo} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import {EProjectName} from 'constants/common';

import {EFooterProject} from 'components/Footer/types';
import {ELoadableReducer} from 'types/common/ELoadableReducer';
import {IBusesBookQuery} from 'types/buses/booking/IBusesBookQuery';

import bookReducer from 'reducers/buses/book/reducer';
import requestBookPageInfoAction from 'reducers/buses/book/thunk/requestBookPageInfoAction';

import rideOfferSelector from 'selectors/buses/book/rideOfferSelector';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import useServerDataFetcher from 'utilities/hooks/useServerDataFetcher';
import {requestBookPageInfoSSR} from 'projects/buses/pages/booking/BookPage/utilities/requestBookPageInfoSSR';
import useQuery from 'utilities/hooks/useQuery';

import withReducers from 'containers/withReducers/withReducers';

import LayoutDefault from 'components/Layouts/LayoutDefault/LayoutDefault';
import BookingLayout from 'components/Layouts/BookingLayout/BookingLayout';
import BookSteps from 'projects/buses/pages/booking/BookPage/components/BookSteps/BookSteps';
import LeftColumn from 'projects/buses/pages/booking/BookPage/components/LeftColumn/LeftColumn';
import RightColumn from 'projects/buses/pages/booking/BookPage/components/RightColumn/RightColumn';
import Form from 'projects/buses/pages/booking/BookPage/components/Form/Form';
import BottomFading from 'components/Skeletons/BottomFading/BottomFading';
import Skeleton from 'projects/buses/pages/booking/BookPage/components/Skeleton/Skeleton';
import ErrorModal from 'projects/buses/pages/booking/BookPage/components/ErrorModal/ErrorModal';

import cx from './BookPage.scss';

const BookPage: FC = () => {
    const needToRender = useServerDataFetcher([requestBookPageInfoSSR]);

    const deviceType = useDeviceType();
    const dispatch = useDispatch();
    const {from, to, rideId, when} = useQuery<keyof IBusesBookQuery>([
        'rideId',
        'from',
        'to',
        'when',
    ]);

    const rideOfferInfo = useSelector(rideOfferSelector);

    useEffect(() => {
        if (!rideId || !from || !to || !when) {
            return;
        }

        dispatch(
            requestBookPageInfoAction({
                params: {
                    rideId,
                    from,
                    to,
                    when,
                },
                isSSR: false,
            }),
        );
    }, [dispatch, from, rideId, to, when]);

    const layout = useMemo(() => {
        if (!rideOfferInfo.isFetched) {
            return <Skeleton className={cx('layout')} />;
        }

        if (rideOfferInfo.isFailed || !rideOfferInfo.value.ride) {
            if (!from || !to || !when) {
                return null;
            }

            return <ErrorModal from={from} to={to} when={when} />;
        }

        return (
            <Form rideOffer={rideOfferInfo.value}>
                <BookingLayout
                    className={cx('layout')}
                    deviceType={deviceType}
                    leftColumn={<LeftColumn rideOffer={rideOfferInfo.value} />}
                    rightColumn={
                        <RightColumn rideOffer={rideOfferInfo.value} />
                    }
                />
            </Form>
        );
    }, [
        deviceType,
        from,
        rideOfferInfo.isFailed,
        rideOfferInfo.isFetched,
        rideOfferInfo.value,
        to,
        when,
    ]);

    const content = useMemo(
        () => (
            <>
                <BookSteps />

                {layout}
            </>
        ),
        [layout],
    );

    if (!needToRender) {
        return null;
    }

    return (
        <LayoutDefault
            className={cx('root', deviceMods('root', deviceType))}
            footerClassName={cx('footer')}
            showSearchForm={deviceType.isDesktop}
            showNavigation={deviceType.isDesktop}
            searchFormInitialIsExpanded={false}
            project={EProjectName.BUSES}
            footerType={EFooterProject.BUSES}
            showFooter={rideOfferInfo.isFetched}
        >
            {content}

            {!rideOfferInfo.isFetched && <BottomFading />}
        </LayoutDefault>
    );
};

export default withReducers([[ELoadableReducer.BUSES_BOOK, bookReducer]])(
    memo(BookPage),
);
