import {FC, ReactNode, useCallback} from 'react';
import arrayMutators from 'final-form-arrays';
import setFieldData from 'final-form-set-field-data';
import {FormRenderProps} from 'react-final-form';
import {useDispatch} from 'react-redux';
import isEmpty from 'lodash/isEmpty';

import {validationInfo} from 'projects/buses/pages/booking/BookPage/components/Form/constants/validationInfo';
import {EFormKey} from 'constants/form/EFormKey';
import {SOLOMON_BUSES_PAY_BUTTON_CLICK} from 'constants/solomon';

import {IBusesBookFormValues} from 'types/buses/booking/IBusesBookFormValues';
import {IBusesCreateRideOfferResponse} from 'server/api/BusesTravelApi/types/TBusesCreateRideOfferApiResponse';
import {EBusesGoal} from 'utilities/metrika/types/goals/buses';

import createOrRestoreOrderAction from 'reducers/buses/book/thunk/createOrRestoreOrderAction/createOrRestoreOrderAction';

import {useBoolean} from 'utilities/hooks/useBoolean';
import {sendPaymentCounter} from 'utilities/solomon';
import {reachGoal} from 'utilities/metrika';

import * as i18n from 'i18n/trains-details-loading';

import CommonBookForm from 'components/Form/Form';
import createSubmitErrorDecorator from 'components/Form/decorators/createSubmitErrorDecorator';
import focusFirstInvalidField from 'components/Form/mutators/focusFirstInvalidField';
import BookLoader from 'components/BookLoader/BookLoader';
import useInitialValues from 'projects/buses/pages/booking/BookPage/components/Form/hooks/useInitialValues';
import ErrorModal from 'projects/buses/pages/booking/BookPage/components/Form/components/ErrorModal/ErrorModal';

const DEFAULT_SUBSCRIPTION = {};

const focusOnSubmitErrors = createSubmitErrorDecorator<IBusesBookFormValues>(
    EFormKey.BUS_BOOK,
    (values, errors) => {
        if (isEmpty(errors)) {
            return;
        }

        reachGoal(EBusesGoal.BOOK_PAGE_PAY_CLICK_VALIDATION_FAILED);
    },
);

const DECORATORS = [focusOnSubmitErrors];

const MUTATORS = {
    ...arrayMutators,
    setFieldData,
    focusFirstInvalidField,
};

interface IFormProps {
    rideOffer: IBusesCreateRideOfferResponse;
}

const Form: FC<IFormProps> = props => {
    const {children, rideOffer} = props;

    const {
        value: isCreatingOrderLoaderVisible,
        setTrue: showCreatingOrderLoader,
        setFalse: hideCreatingOrderLoader,
    } = useBoolean(false);

    const {
        value: isCreatingOrderErrorVisible,
        setTrue: showCreatingOrderError,
    } = useBoolean(false);

    const dispatch = useDispatch();
    const initialValues = useInitialValues(rideOffer);

    const {citizenships} = rideOffer.bookParams;

    const handleCreatingOrderError = useCallback(() => {
        hideCreatingOrderLoader();
        showCreatingOrderError();

        reachGoal(EBusesGoal.BOOK_PAGE_CREATE_ORDER_ERROR);
    }, [hideCreatingOrderLoader, showCreatingOrderError]);

    const handleFormSubmit = useCallback(
        (formData: IBusesBookFormValues) => {
            showCreatingOrderLoader();
            sendPaymentCounter(SOLOMON_BUSES_PAY_BUTTON_CLICK);

            dispatch(
                createOrRestoreOrderAction({
                    params: {
                        formData,
                        citizenships,
                        label: rideOffer.label,
                        onError: handleCreatingOrderError,
                    },
                }),
            );
        },
        [
            citizenships,
            rideOffer.label,
            dispatch,
            handleCreatingOrderError,
            showCreatingOrderLoader,
        ],
    );

    const render = useCallback(
        ({handleSubmit}: FormRenderProps): ReactNode => {
            return (
                <form
                    name={EFormKey.BUS_BOOK}
                    autoComplete="off"
                    noValidate
                    onSubmit={handleSubmit}
                >
                    {children}
                </form>
            );
        },
        [children],
    );

    return (
        <>
            <CommonBookForm<IBusesBookFormValues>
                validationInfo={validationInfo}
                subscription={DEFAULT_SUBSCRIPTION}
                initialValues={initialValues}
                decorators={DECORATORS}
                mutators={MUTATORS}
                render={render}
                onSubmit={handleFormSubmit}
            />

            <BookLoader
                isModalView
                isLoading={isCreatingOrderLoaderVisible}
                title={i18n.purchaseDashPageDashReservingDashTickets()}
            />

            <ErrorModal isVisible={isCreatingOrderErrorVisible} />
        </>
    );
};

export default Form;
