import {useMemo} from 'react';
import times from 'lodash/times';
import first from 'lodash/first';

import {EFormKey} from 'constants/form/EFormKey';
import {EFieldName} from 'components/BookingPassengerForm/constants/fieldNames';
import {DEFAULT_PASSENGER} from 'projects/buses/pages/booking/BookPage/components/Form/constants';

import {
    EBusesPassengerFormFiled,
    IBusesPassenger,
} from 'types/buses/booking/IBusesBookFormValues';
import {IBusesCreateRideOfferResponse} from 'server/api/BusesTravelApi/types/TBusesCreateRideOfferApiResponse';
import {IBusesBookQuery} from 'types/buses/booking/IBusesBookQuery';

import {TUserInfo} from 'reducers/common/commonReducerTypes';

import {getBookingPassengers} from 'utilities/localStorage/booking/getBookingPassengers';
import getBaggagePrice from 'projects/buses/pages/booking/BookPage/utilities/getBaggagePrice';
import getTariffs from 'projects/buses/pages/booking/BookPage/utilities/getTariffs';
import {getQueryByBrowserHistory} from 'utilities/getQueryByBrowserHistory/getQueryByBrowserHistory';
import getDefaultDocumentType from 'projects/buses/pages/booking/BookPage/components/Form/hooks/utilities/getDefaultDocumentType';
import getDefaultCitizenship from 'projects/buses/pages/booking/BookPage/components/Form/hooks/utilities/getDefaultCitizenship';

export default function useStoredValidatedPassengers({
    userInfo,
    rideOffer,
}: {
    userInfo: TUserInfo;
    rideOffer: IBusesCreateRideOfferResponse;
}): IBusesPassenger[] {
    const baggagePrice = useMemo(
        () => getBaggagePrice(rideOffer.bookParams.ticketTypes),
        [rideOffer.bookParams.ticketTypes],
    );

    const tariffs = useMemo(
        () => getTariffs(rideOffer.bookParams.ticketTypes),
        [rideOffer.bookParams.ticketTypes],
    );

    const allowedDocuments = rideOffer.bookParams.documentTypes;
    const {citizenships} = rideOffer.bookParams;

    const storedPassengers = useMemo(
        () =>
            getBookingPassengers<IBusesPassenger>(EFormKey.BUS_BOOK, userInfo),
        [userInfo],
    );

    return useMemo(() => {
        const validatedPassengers = storedPassengers.map(rawStoredPassenger => {
            const storedPassenger = {...rawStoredPassenger};
            const storedDocumentType = storedPassenger[EFieldName.documentType];

            if (
                storedDocumentType &&
                !allowedDocuments.includes(storedDocumentType)
            ) {
                storedPassenger[EFieldName.documentType] =
                    getDefaultDocumentType(allowedDocuments);
                storedPassenger[EFieldName.citizenship] =
                    getDefaultCitizenship(citizenships);

                delete storedPassenger[EFieldName.documentNumber];
                delete storedPassenger[EFieldName.documentValidDate];
            }

            const storedTariff = storedPassenger.tariff;

            if (storedTariff && !tariffs.includes(storedTariff)) {
                storedPassenger.tariff = first(tariffs);
            }

            if (!baggagePrice) {
                delete storedPassenger[EBusesPassengerFormFiled.BAGGAGE_COUNT];
            }

            return storedPassenger;
        });

        const {placesCount}: Partial<IBusesBookQuery> =
            getQueryByBrowserHistory();

        return times(
            Number(placesCount),
            index => validatedPassengers[index] || DEFAULT_PASSENGER,
        );
    }, [
        allowedDocuments,
        baggagePrice,
        citizenships,
        storedPassengers,
        tariffs,
    ]);
}
