import {FC, Fragment, memo, useMemo} from 'react';
import {useSelector} from 'react-redux';

import {
    EBusesBookGroup,
    TBusesPassengerTicketTariff,
} from 'types/buses/booking/IBusesBookFormValues';
import {ICountry} from 'types/common/ICountry';
import {IBusesBookQuery} from 'types/buses/booking/IBusesBookQuery';
import IPrice from 'types/common/price/IPrice';

import {getUserInfo} from 'selectors/common/userInfoSelector';
import travelerPassengersSelector from 'selectors/common/passengers/travelerPassengersSelector';

import {
    getQa,
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import useQuery from 'utilities/hooks/useQuery';
import getPassengersAndBaggageCount from 'projects/buses/pages/booking/BookPage/components/LeftColumn/components/Passengers/utilities/getPassengersAndBaggageCount';
import getAvailableBaggageCount from 'projects/buses/pages/booking/BookPage/components/LeftColumn/components/Passengers/utilities/getAvailableBaggageCount';
import getParsedQueryPlaces from 'projects/buses/pages/booking/BookPage/utilities/getParsedQueryPlaces';

import * as i18n from 'i18n/buses-BookPage-Passengers';

import Form from 'components/Form/Form';
import Separator from 'components/Separator/Separator';
import Passenger from 'projects/buses/pages/booking/BookPage/components/LeftColumn/components/Passengers/components/Passenger/Passenger';
import BookingLayout from 'components/Layouts/BookingLayout/BookingLayout';
import Box from 'components/Box/Box';
import usePassengers from 'projects/buses/pages/booking/BookPage/components/LeftColumn/components/Passengers/hooks/usePassengers';

import {EDocumentType} from 'server/api/TravelersApi/enums/EDocumentType';

export interface IPassengersProps extends IWithQaAttributes {
    countries: ICountry[];
    ticketLimit: number;
    allowedDocuments: EDocumentType[];
    tariffs: TBusesPassengerTicketTariff[];
    baggagePrice: IPrice | null;
}

const Passengers: FC<IPassengersProps> = props => {
    const {countries, ticketLimit, allowedDocuments, tariffs, baggagePrice} =
        props;

    const rootQa = getQa(props);

    const deviceType = useDeviceType();
    const travelerPassengers = useSelector(travelerPassengersSelector);
    const userInfo = useSelector(getUserInfo);

    const {placesCount: placesCountRaw, places: rawPlaces} = useQuery<
        keyof IBusesBookQuery
    >(['placesCount', 'places']);

    const placesCount = Number(placesCountRaw);

    const places = useMemo(
        () => getParsedQueryPlaces(rawPlaces || undefined),
        [rawPlaces],
    );

    const passengers = usePassengers({
        userInfo,
        count: placesCount,
    });

    const passengersAndBaggageCount = getPassengersAndBaggageCount(passengers);

    const passengersBlock = useMemo(() => {
        return (
            <Box between={deviceType.isMobile ? 5 : 0}>
                <Form.FieldGroup groupId={EBusesBookGroup.PASSENGERS}>
                    {passengers.map((passenger, index) => {
                        const availableBaggageCount = getAvailableBaggageCount(
                            ticketLimit,
                            passengersAndBaggageCount,
                            passenger,
                        );

                        const passengerForm = (
                            <Passenger
                                title={i18n.passengerTitle({
                                    index: index + 1,
                                    place: places?.[index],
                                })}
                                initialValues={passenger}
                                formGroupId={index}
                                travelerPassengers={travelerPassengers}
                                countries={countries}
                                allowedDocuments={allowedDocuments}
                                tariffs={tariffs}
                                baggagePrice={baggagePrice}
                                availableBaggageCount={availableBaggageCount}
                                maxBaggageCount={ticketLimit - 1}
                                ticketLimit={ticketLimit}
                                {...prepareQaAttributes({
                                    parent: rootQa,
                                    current: 'passenger',
                                })}
                            />
                        );

                        if (deviceType.isMobile) {
                            return (
                                <BookingLayout.Card key={index}>
                                    {passengerForm}
                                </BookingLayout.Card>
                            );
                        }

                        return (
                            <Fragment key={index}>
                                {index > 0 && <Separator margin={8} />}

                                {passengerForm}
                            </Fragment>
                        );
                    })}
                </Form.FieldGroup>
            </Box>
        );
    }, [
        deviceType.isMobile,
        passengers,
        ticketLimit,
        passengersAndBaggageCount,
        places,
        travelerPassengers,
        countries,
        allowedDocuments,
        tariffs,
        baggagePrice,
        rootQa,
    ]);

    if (deviceType.isMobile) {
        return <div>{passengersBlock}</div>;
    }

    return <BookingLayout.Card>{passengersBlock}</BookingLayout.Card>;
};

export default memo(Passengers);
