import {memo, FC} from 'react';

import {
    IBusesBookFormValues,
    IBusesPassenger,
    TBusesPassengerTicketTariff,
} from 'types/buses/booking/IBusesBookFormValues';
import {IPassengerWithDocumentsDTO} from 'server/api/TravelersApi/types/IPassengerDTO';
import {ICountry} from 'types/common/ICountry';
import IPrice from 'types/common/price/IPrice';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {deviceMods} from 'utilities/stylesUtils';

import Form from 'components/Form/Form';
import BookingPassengerForm from 'components/BookingPassengerForm/BookingPassengerForm';
import TariffAndBaggage from 'projects/buses/pages/booking/BookPage/components/LeftColumn/components/Passengers/components/Passenger/components/TariffAndBaggage/TariffAndBaggage';

import {EDocumentType} from 'server/api/TravelersApi/enums/EDocumentType';

import cx from './Passenger.scss';

interface IPassengerProps extends IWithQaAttributes {
    title: string;
    initialValues: IBusesPassenger;
    formGroupId: number;
    travelerPassengers: IPassengerWithDocumentsDTO[];
    countries: ICountry[];
    allowedDocuments: EDocumentType[];
    tariffs: TBusesPassengerTicketTariff[];
    maxBaggageCount: number;
    availableBaggageCount: number;
    baggagePrice: IPrice | null;
    ticketLimit: number;
}

const Passenger: FC<IPassengerProps> = props => {
    const {
        title,
        initialValues,
        formGroupId,
        travelerPassengers,
        countries,
        allowedDocuments,
        tariffs,
        maxBaggageCount,
        availableBaggageCount,
        baggagePrice,
        ticketLimit,
        ...rest
    } = props;

    const deviceType = useDeviceType();

    return (
        <div
            className={cx('root', deviceMods('root', deviceType))}
            {...prepareQaAttributes(rest)}
        >
            <Form.FieldGroup groupId={formGroupId}>
                <BookingPassengerForm<IBusesBookFormValues>
                    initialValues={initialValues}
                    deviceType={deviceType}
                    title={title}
                    countries={countries}
                    passengers={travelerPassengers}
                    availableDocuments={allowedDocuments}
                    excludeRuCitizenForOtherDocument
                    hideDisabledCitizenship
                    hasPatronymic
                    {...prepareQaAttributes({
                        parent: rest,
                        current: 'form',
                    })}
                />

                <TariffAndBaggage
                    className={cx('tariffAndBaggage')}
                    tariffs={tariffs}
                    maxBaggageCount={maxBaggageCount}
                    availableBaggageCount={availableBaggageCount}
                    baggagePrice={baggagePrice}
                    ticketLimit={ticketLimit}
                />
            </Form.FieldGroup>
        </div>
    );
};

export default memo(Passenger);
