import {memo, FC, useCallback, useMemo} from 'react';
import times from 'lodash/times';

import {IWithClassName} from 'types/withClassName';
import {EBusesPassengerFormFiled} from 'types/buses/booking/IBusesBookFormValues';
import IPrice from 'types/common/price/IPrice';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';

import * as i18n from 'i18n/buses-BookPage-Passengers';

import FormField from 'components/FormField/FormField';
import {TControlRenderFunc} from 'components/FormField/components/Field/Field';
import RadioButton from 'components/RadioButton/RadioButton';
import Flex from 'components/Flex/Flex';
import Hint from 'projects/buses/pages/booking/BookPage/components/LeftColumn/components/Passengers/components/Passenger/components/TariffAndBaggage/components/Baggage/components/Hint/Hint';
import OptionText from 'projects/buses/pages/booking/BookPage/components/LeftColumn/components/Passengers/components/Passenger/components/TariffAndBaggage/components/Baggage/components/OptionText/OptionText';

import cx from './Baggage.scss';

interface IBaggageProps extends IWithClassName {
    maxCount: number;
    availableCount: number;
    price: IPrice;
    ticketLimit: number;
}

const Baggage: FC<IBaggageProps> = props => {
    const {className, maxCount, availableCount, price, ticketLimit} = props;

    const deviceType = useDeviceType();

    const control: TControlRenderFunc = useCallback(
        ({input}) => {
            return (
                <RadioButton
                    {...input}
                    size="l"
                    width="max"
                    data-name={input.name}
                >
                    {times(maxCount, count => (
                        <RadioButton.Radio
                            key={count}
                            value={count}
                            disabled={count > availableCount}
                        >
                            <OptionText
                                className={cx('optionText')}
                                count={count}
                                price={price}
                            />
                        </RadioButton.Radio>
                    ))}
                </RadioButton>
            );
        },
        [availableCount, maxCount, price],
    );

    const title = useMemo(
        () => (
            <Flex alignItems="center" between={1} inline nowrap>
                <span>{i18n.baggageTitle()}</span>

                <Hint ticketLimit={ticketLimit} />
            </Flex>
        ),
        [ticketLimit],
    );

    return (
        <FormField
            className={cx(deviceMods('root', deviceType), className)}
            name={EBusesPassengerFormFiled.BAGGAGE_COUNT}
            title={title}
            deviceType={deviceType}
            control={control}
            initialValue="0"
        />
    );
};

export default memo(Baggage);
