import {useEffect} from 'react';
import {useForm} from 'react-final-form';
import times from 'lodash/times';
import debounce from 'lodash/debounce';

import {EFormKey} from 'constants/form/EFormKey';
import {DEFAULT_PASSENGER} from 'projects/buses/pages/booking/BookPage/components/Form/constants';

import {
    EBusesBookGroup,
    IBusesBookFormValues,
    IBusesPassenger,
} from 'types/buses/booking/IBusesBookFormValues';

import {TUserInfo} from 'reducers/common/commonReducerTypes';

import {saveBookingPassengers} from 'utilities/localStorage/booking/saveBookingPassengers';
import {usePrevious} from 'utilities/hooks/usePrevious';

import useMemoFormFieldValue from 'components/Form/hooks/useMemoFormFieldValue';

const saveBookingPassengersDebounced = debounce(saveBookingPassengers, 2000);

export default function usePassengers({
    userInfo,
    count,
}: {
    userInfo: TUserInfo;
    count: number;
}): IBusesPassenger[] {
    const prevCount = usePrevious(count);
    const form = useForm<IBusesBookFormValues>();
    const passengersFormData = useMemoFormFieldValue<IBusesPassenger[]>(
        EBusesBookGroup.PASSENGERS,
    );

    useEffect(() => {
        if (prevCount === count) {
            return;
        }

        const updatedPassengers = times(
            count,
            index => passengersFormData[index] || DEFAULT_PASSENGER,
        );

        form.change(EBusesBookGroup.PASSENGERS, updatedPassengers);
    }, [count, form, passengersFormData, prevCount]);

    useEffect(() => {
        if (!passengersFormData.length) {
            return;
        }

        saveBookingPassengersDebounced(
            EFormKey.BUS_BOOK,
            passengersFormData,
            userInfo,
        );
    }, [passengersFormData, userInfo]);

    return passengersFormData;
}
