import {FC, memo, useEffect, useMemo} from 'react';

import {IBusesPlace} from 'types/buses/booking/IBusesBookParams';
import {IWithClassName} from 'types/withClassName';
import {IBusesBookQuery} from 'types/buses/booking/IBusesBookQuery';

import {PLACES_URL_SEPARATOR} from 'projects/buses/utilities/urls/getBookUrl';
import useQuery from 'utilities/hooks/useQuery';
import useImmutableCallback from 'utilities/hooks/useImmutableCallback';
import updateBookUrlQuery from 'projects/buses/utilities/urls/updateBookUrlQuery';
import getSchemaCells from 'projects/buses/pages/booking/BookPage/components/LeftColumn/components/Places/components/SchemaInfo/components/Schema/utilities/getSchemaCells';
import getNewPlaces from 'projects/buses/pages/booking/BookPage/components/LeftColumn/components/Places/components/SchemaInfo/components/Schema/utilities/getNewPlaces';
import getDriverPlace from 'projects/buses/pages/booking/BookPage/components/LeftColumn/components/Places/components/SchemaInfo/components/Schema/utilities/getDriverPlace';
import getParsedQueryPlaces from 'projects/buses/pages/booking/BookPage/utilities/getParsedQueryPlaces';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import isLongBus from 'projects/buses/pages/booking/BookPage/components/LeftColumn/components/Places/components/SchemaInfo/components/Schema/utilities/isLongBus';

import * as i18n from 'i18n/buses-BookPage-Places';

import Card from 'components/Card/Card';
import Flex from 'components/Flex/Flex';
import Box from 'components/Box/Box';
import WheelIcon from 'icons/32/Wheel';
import HorizontalScroller from 'components/HorizontalScroller/HorizontalScroller';
import Cell from 'projects/buses/pages/booking/BookPage/components/LeftColumn/components/Places/components/SchemaInfo/components/Schema/components/Cell/Cell';
import MessageBoxPopup from 'components/MessageBoxPopup/MessageBoxPopup';
import usePlaceHint from 'projects/buses/pages/booking/BookPage/components/LeftColumn/components/Places/components/SchemaInfo/components/Schema/hooks/usePlaceHint';

import cx from './Schema.scss';

interface ISchemaProps extends IWithClassName {
    seats: string[];
    placesMap: IBusesPlace[];
}

const Schema: FC<ISchemaProps> = props => {
    const {className, seats, placesMap} = props;

    const deviceType = useDeviceType();

    const {places, placesCount: placesCountRaw} = useQuery<
        keyof IBusesBookQuery
    >(['places', 'placesCount']);

    const placesCount = Number(placesCountRaw);
    const placeHint = usePlaceHint();

    const selectedPlaces = useMemo(
        () => getParsedQueryPlaces(places || undefined),
        [places],
    );

    const selectedPlacesSet = useMemo(
        () => new Set(selectedPlaces),
        [selectedPlaces],
    );

    const driverPlace = useMemo(() => getDriverPlace(placesMap), [placesMap]);

    const schemaCells = useMemo(
        () => getSchemaCells(placesMap, seats, driverPlace),
        [driverPlace, placesMap, seats],
    );
    const busIsLong = useMemo(() => isLongBus(schemaCells), [schemaCells]);

    const handlePlaceClick = useImmutableCallback((newPlace: string) => {
        const newPlaces = getNewPlaces(selectedPlaces, newPlace, placesCount);

        updateBookUrlQuery({
            places: newPlaces.join(PLACES_URL_SEPARATOR) || undefined,
        });
    });

    useEffect(() => {
        if (placesCount < selectedPlaces.length) {
            updateBookUrlQuery({
                places:
                    selectedPlaces
                        .slice(0, placesCount)
                        .join(PLACES_URL_SEPARATOR) || undefined,
            });
        }
    }, [placesCount, selectedPlaces]);

    if (!schemaCells) {
        return null;
    }

    return (
        <>
            <HorizontalScroller offset={4}>
                <Card
                    className={cx('root', className)}
                    border="normal"
                    x={5}
                    y={5}
                    inline
                    between={4}
                    nowrap
                >
                    {!driverPlace && <WheelIcon className={cx('wheel')} />}

                    <Box between={2}>
                        {schemaCells.map((row, index) => {
                            return (
                                <Flex key={index} between={2} inline nowrap>
                                    {row.map((cell, rowIndex) => (
                                        <Cell
                                            key={rowIndex}
                                            cell={cell}
                                            isSelected={selectedPlacesSet.has(
                                                cell.id,
                                            )}
                                            isSmall={
                                                deviceType.isDesktop &&
                                                busIsLong
                                            }
                                            onClick={handlePlaceClick}
                                            onShowHint={placeHint.show}
                                            onHideHint={placeHint.hide}
                                        />
                                    ))}
                                </Flex>
                            );
                        })}
                    </Box>
                </Card>
            </HorizontalScroller>

            <MessageBoxPopup
                className={cx('placeHint')}
                isVisible={placeHint.isVisible}
                anchorRef={placeHint.anchorRef}
                hasClose={false}
            >
                {i18n.placeHintOccupied()}
            </MessageBoxPopup>
        </>
    );
};

export default memo(Schema);
