import {FC, memo, RefObject, useCallback, useRef} from 'react';

import {
    EBusesPlaceStatus,
    EBusesPlaceType,
} from 'types/buses/booking/IBusesBookParams';
import {ISchemaCell} from 'projects/buses/pages/booking/BookPage/components/LeftColumn/components/Places/components/SchemaInfo/components/Schema/types';

import WheelIcon from 'icons/32/Wheel';

import cx from './Cell.scss';

interface ICellProps {
    cell: ISchemaCell;
    isSelected: boolean;
    isSmall: boolean;
    onClick(place: string): void;
    onShowHint(ref: RefObject<HTMLDivElement>): void;
    onHideHint(ref: RefObject<HTMLDivElement>): void;
}

const Cell: FC<ICellProps> = props => {
    const {cell, isSelected, isSmall, onClick, onShowHint, onHideHint} = props;

    const rootRef = useRef<HTMLDivElement | null>(null);

    const isSeat = cell.type === EBusesPlaceType.SEAT;
    const isFreeSeat = isSeat && cell.status === EBusesPlaceStatus.FREE;
    const isOccupiedSeat = isSeat && cell.status === EBusesPlaceStatus.OCCUPIED;

    const handleClick = useCallback(() => {
        onClick(cell.id);
    }, [cell.id, onClick]);

    const handleMouseEnter = useCallback(() => {
        onShowHint(rootRef);
    }, [onShowHint]);

    const handleMouseLeave = useCallback(() => {
        onHideHint(rootRef);
    }, [onHideHint]);

    if (cell.type === EBusesPlaceType.DRIVER) {
        return <WheelIcon className={cx('wheel')} />;
    }

    return (
        <div
            className={cx('root', {
                root_freeSeat: isFreeSeat,
                root_selected: isSelected,
                root_occupied: cell.status === EBusesPlaceStatus.OCCUPIED,
                root_small: isSmall,
            })}
            ref={rootRef}
            onClick={isFreeSeat ? handleClick : undefined}
            onMouseEnter={isOccupiedSeat ? handleMouseEnter : undefined}
            onMouseLeave={isOccupiedSeat ? handleMouseLeave : undefined}
        >
            {isSeat ? cell.id : null}
        </div>
    );
};

export default memo(Cell);
