import {memo, FC, useCallback, useMemo} from 'react';

import {IBusesBookQuery} from 'types/buses/booking/IBusesBookQuery';
import {IWithClassName} from 'types/withClassName';

import useQuery from 'utilities/hooks/useQuery';
import updateBookUrlQuery from 'projects/buses/utilities/urls/updateBookUrlQuery';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import getMaxPlacesHintText from 'projects/buses/pages/booking/BookPage/components/LeftColumn/components/Places/components/Selector/utilities/getMaxPlacesHintText';

import * as i18n from 'i18n/buses-BookPage-Places';

import TravellersCount from 'components/TravellersCount/TravellersCount';
import Flex from 'components/Flex/Flex';
import Box from 'components/Box/Box';
import Text from 'components/Text/Text';
import useBaggagePlacesCount from 'projects/buses/pages/booking/BookPage/components/LeftColumn/components/Places/components/Selector/hooks/useBaggagePlacesCount';

import cx from './Selector.scss';

interface ISelectorProps extends IWithClassName {
    ticketLimit: number;
    freeSeats: number;
}

const Selector: FC<ISelectorProps> = props => {
    const {className, ticketLimit, freeSeats} = props;

    const deviceType = useDeviceType();

    const {placesCount: placesCountRaw} = useQuery<keyof IBusesBookQuery>([
        'placesCount',
    ]);

    const baggagePlacesCount = useBaggagePlacesCount();
    const placesCount = Number(placesCountRaw);

    const maxPlacesCount = Math.min(
        freeSeats,
        ticketLimit - baggagePlacesCount,
    );

    const maxPlacesHintText = useMemo(
        () =>
            getMaxPlacesHintText({
                placesCount,
                ticketLimit,
                freeSeats,
                baggagePlacesCount,
            }),
        [baggagePlacesCount, freeSeats, placesCount, ticketLimit],
    );

    const maxPlacesHint = useMemo(() => {
        if (!maxPlacesHintText) {
            return null;
        }

        if (deviceType.isMobile) {
            return (
                <Box between={4}>
                    <Text size="l" weight="bold" tag="div">
                        {i18n.title()}
                    </Text>

                    <div>{maxPlacesHintText}</div>
                </Box>
            );
        }

        return maxPlacesHintText;
    }, [deviceType.isMobile, maxPlacesHintText]);

    const handleChangePlacesCount = useCallback((newPlacesCount: number) => {
        updateBookUrlQuery({placesCount: String(newPlacesCount)});
    }, []);

    return (
        <Flex
            className={cx(deviceMods('root', deviceType), className)}
            flexDirection={deviceType.isMobile ? 'row' : 'row-reverse'}
            justifyContent={deviceType.isMobile ? 'flex-start' : 'flex-end'}
            alignItems="center"
        >
            <div className={cx('description')}>{i18n.counterDescription()}</div>

            <TravellersCount
                className={cx('counter')}
                min={1}
                max={maxPlacesCount}
                value={placesCount}
                countWidth={34}
                maxCountHint={maxPlacesHint}
                onChange={handleChangePlacesCount}
            />
        </Flex>
    );
};

export default memo(Selector);
