import {memo, FC, useMemo} from 'react';

import {
    EBusesBookGroup,
    IBusesPassenger,
} from 'types/buses/booking/IBusesBookFormValues';
import {IBusesTicketType} from 'types/buses/booking/IBusesBookParams';
import {EBusesGoal} from 'utilities/metrika/types/goals/buses';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import getTicketTypeMap from 'projects/buses/pages/booking/BookPage/components/RightColumn/components/Cart/utilities/getTicketTypeMap';
import getTickets from 'projects/buses/pages/booking/BookPage/components/RightColumn/components/Cart/utilities/getTickets';
import getTotalPrice from 'projects/buses/pages/booking/BookPage/components/RightColumn/components/Cart/utilities/getTotalPrice';
import getTotalYandexFee from 'projects/buses/pages/booking/BookPage/components/RightColumn/components/Cart/utilities/getTotalYandexFee';
import {useReachGoal} from 'utilities/metrika/useReachGoal';

import * as i18n from 'i18n/buses-BookPage-Cart';

import BookingLayout from 'components/Layouts/BookingLayout/BookingLayout';
import Text from 'components/Text/Text';
import Separator from 'components/Separator/Separator';
import Button from 'components/Button/Button';
import useMemoFormFieldValue from 'components/Form/hooks/useMemoFormFieldValue';
import Tickets from 'projects/buses/pages/booking/BookPage/components/RightColumn/components/Cart/components/Tickets/Tickets';
import TotalPrice from 'projects/buses/pages/booking/BookPage/components/RightColumn/components/Cart/components/TotalPrice/TotalPrice';

import cx from './Cart.scss';

interface ICartProps {
    ticketTypes: IBusesTicketType[];
}

const Cart: FC<ICartProps> = props => {
    const {ticketTypes} = props;

    const deviceType = useDeviceType();

    const passengers = useMemoFormFieldValue<IBusesPassenger[]>(
        EBusesBookGroup.PASSENGERS,
    );

    const ticketTypeMap = useMemo(
        () => getTicketTypeMap(ticketTypes),
        [ticketTypes],
    );

    const tickets = useMemo(
        () => getTickets(passengers, ticketTypeMap),
        [passengers, ticketTypeMap],
    );

    const totalPrice = useMemo(() => getTotalPrice(tickets), [tickets]);
    const yandexFee = useMemo(() => getTotalYandexFee(tickets), [tickets]);

    const handlePayClick = useReachGoal(EBusesGoal.BOOK_PAGE_PAY_CLICK);

    return (
        <BookingLayout.Cart className={cx(deviceMods('root', deviceType))}>
            <Text size="l" weight="bold" tag="div">
                {i18n.title()}
            </Text>

            <Tickets className={cx('tickets')} tickets={tickets} />

            <Separator margin={5} />

            {totalPrice && yandexFee && (
                <TotalPrice
                    className={cx('totalPrice')}
                    price={totalPrice}
                    fee={yandexFee}
                />
            )}

            <Button
                className={cx('payButton')}
                theme="primary"
                size={deviceType.isMobile ? 'l' : 'xl'}
                type="submit"
                width="max"
                onClick={handlePayClick}
            >
                {i18n.payButton()}
            </Button>
        </BookingLayout.Cart>
    );
};

export default memo(Cart);
