import {memo, FC, useMemo} from 'react';

import IPrice from 'types/common/price/IPrice';
import {IWithClassName} from 'types/withClassName';

import {useDeviceType} from 'utilities/hooks/useDeviceType';

import * as i18n from 'i18n/buses-BookPage-Cart';

import Flex from 'components/Flex/Flex';
import Price from 'components/Price/Price';
import InfoPopup from 'components/InfoPopup/InfoPopup';
import Text from 'components/Text/Text';
import Box from 'components/Box/Box';

import cx from './TotalPrice.scss';

interface ITotalPriceProps extends IWithClassName {
    price: IPrice;
    fee: IPrice;
}

const TotalPrice: FC<ITotalPriceProps> = props => {
    const {className, price, fee} = props;

    const deviceType = useDeviceType();

    const ticketsPrice = useMemo((): IPrice => {
        return {
            ...price,
            value: price.value - fee.value,
        };
    }, [fee.value, price]);

    const hintContent = useMemo(() => {
        const content = (
            <div>
                <Flex justifyContent="space-between">
                    <div>{i18n.totalPriceHintTicketsPrice()}</div>
                    <Price {...ticketsPrice} />
                </Flex>

                <Flex justifyContent="space-between" above={1}>
                    <div>{i18n.totalPriceHintFee()}</div>
                    <Price {...fee} />
                </Flex>
            </div>
        );

        if (deviceType.isMobile) {
            return (
                <Box between={4}>
                    <Text size="l" weight="bold" tag="div">
                        {i18n.totalPriceHintTitle()}
                    </Text>

                    {content}
                </Box>
            );
        }

        return content;
    }, [deviceType.isMobile, fee, ticketsPrice]);

    return (
        <Flex
            className={className}
            justifyContent="space-between"
            textSize="xl"
            textWeight="bold"
        >
            <span>{i18n.totalPrice()}</span>

            <div>
                <Price {...price} />

                <InfoPopup className={cx('icon')}>{hintContent}</InfoPopup>
            </div>
        </Flex>
    );
};

export default memo(TotalPrice);
