import {FC, memo, useCallback, useEffect, useMemo, useState} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import {
    SOLOMON_BUSES_PAYMENT_PAGE_LOADED,
    SOLOMON_BUSES_TRUST_ERROR,
    SOLOMON_BUSES_TRUST_LOADED,
    SOLOMON_BUSES_TRUST_SUCCESS,
} from 'constants/solomon';

import {ESubscriptionCode} from 'types/subscription/ESubscriptionCode';
import {ESubscriptionSource} from 'types/subscription/ESubscriptionSource';
import {ESubscriptionVerticalName} from 'types/subscription/ESubscriptionVerticalName';
import {IBusesBookPaymentQuery} from 'types/buses/booking/IBusesBookPaymentQuery';
import {ELoadableReducer} from 'types/common/ELoadableReducer';
import EGenericOrderState from 'server/api/GenericOrderApi/types/common/EGenericOrderState';
import {EBusesGoal} from 'utilities/metrika/types/goals/buses';

import startBookFlowAction from 'reducers/buses/book/thunk/startBookFlowAction/startBookFlowAction';
import bookReducer from 'reducers/buses/book/reducer';
import stopOrderPolling from 'reducers/buses/book/thunk/stopOrderPolling';

import orderSelector from 'selectors/buses/book/orderSelector';
import orderErrorSelector from 'selectors/buses/book/orderErrorSelector';

import useQuery from 'utilities/hooks/useQuery';
import {usePrevious} from 'utilities/hooks/usePrevious';
import getBusService from 'projects/buses/utilities/booking/getBusService';
import {sendPaymentCounter} from 'utilities/solomon';
import {reachGoal} from 'utilities/metrika';

import BookingMailSubscription from 'containers/BookingMailSubscription/BookingMailSubscription';
import withReducers from 'containers/withReducers/withReducers';

import PaymentLayout from 'components/Layouts/NewPaymentLayout/PaymentLayout';
import PaymentIframe from 'components/PaymentIframe/PaymentIframe';
import ErrorModal from 'projects/buses/pages/booking/BookPaymentPage/components/ErrorModal/ErrorModal';
import OrderInfo from 'projects/buses/pages/booking/BookPaymentPage/components/OrderInfo/OrderInfo';

import useOwnLoader from 'projects/buses/pages/booking/BookPaymentPage/hooks/useOwnLoader';

const BookPaymentPage: FC = () => {
    const [isFrameLoaded, setIsFrameLoaded] = useState(false);
    const [isPaymentStarted, setIsPaymentStarted] = useState(false);
    const [isPaymentCompleted, setIsPaymentCompleted] = useState(false);
    const [isTrustPaymentError, setIsTrustPaymentError] = useState(false);

    const {orderId} = useQuery<keyof IBusesBookPaymentQuery>(['orderId']);
    const order = useSelector(orderSelector);
    const orderError = useSelector(orderErrorSelector);

    const prettyId = order?.prettyId;
    const paymentUrl = order?.payment?.paymentUrl;
    const orderState = order?.state;
    const email = order?.contactInfo.email;
    const isPaymentFailed = orderState === EGenericOrderState.PAYMENT_FAILED;

    const prevPaymentUrl = usePrevious(paymentUrl);

    const dispatch = useDispatch();

    const busService = useMemo(
        () => getBusService(order?.services || []),
        [order?.services],
    );

    const ownLoader = useOwnLoader({
        isFrameLoaded,
        isPaymentCompleted,
        isTrustPaymentError,
        isPaymentFailed,
    });

    const handleFrameLoaded = useCallback(() => {
        setIsFrameLoaded(true);
        reachGoal(EBusesGoal.PAYMENT_IFRAME_LOADED);
    }, []);

    const handlePaymentStart = useCallback(() => {
        setIsPaymentStarted(true);
    }, []);

    const handlePaymentSuccess = useCallback(() => {
        setIsPaymentCompleted(true);
        reachGoal(EBusesGoal.PAYMENT_FINISHED_WITH_SUCCESS);
    }, []);

    const handlePaymentError = useCallback(() => {
        setIsTrustPaymentError(true);
        reachGoal(EBusesGoal.PAYMENT_ERROR);
    }, []);

    useEffect(() => {
        /**
         * Url меняется при старте новой оплаты
         */
        if (prevPaymentUrl !== paymentUrl) {
            setIsFrameLoaded(false);
            setIsTrustPaymentError(false);
        }
    }, [paymentUrl, prevPaymentUrl]);

    useEffect(() => {
        if (!orderId) {
            return;
        }

        dispatch(startBookFlowAction({orderId}));

        return (): void => {
            dispatch(stopOrderPolling());
        };
    }, [dispatch, orderId]);

    useEffect(() => {
        sendPaymentCounter(SOLOMON_BUSES_PAYMENT_PAGE_LOADED);
        reachGoal(EBusesGoal.PAYMENT_PAGE_LOADED);
    }, []);

    const footer = useMemo(() => {
        if (!email) {
            return null;
        }

        return (
            <BookingMailSubscription
                email={email}
                source={ESubscriptionSource.PAYMENT}
                promoSubscriptionCode={ESubscriptionCode.travelNews}
                travelVerticalName={ESubscriptionVerticalName.Bus}
                subscriptionIsAvailable={isPaymentStarted}
            />
        );
    }, [email, isPaymentStarted]);

    const orderInfo = useMemo(() => {
        if (!prettyId || !busService) {
            return null;
        }

        return (
            <OrderInfo prettyId={prettyId} serviceInfo={busService.busInfo} />
        );
    }, [busService, prettyId]);

    const iframe = useMemo(() => {
        if (!paymentUrl) {
            return null;
        }

        return (
            <PaymentIframe
                isNewTrust
                loadEvent={SOLOMON_BUSES_TRUST_LOADED}
                successEvent={SOLOMON_BUSES_TRUST_SUCCESS}
                errorEvent={SOLOMON_BUSES_TRUST_ERROR}
                paymentUrl={paymentUrl}
                onLoad={handleFrameLoaded}
                onStartPayment={handlePaymentStart}
                onPaymentSuccess={handlePaymentSuccess}
                onPaymentError={handlePaymentError}
            />
        );
    }, [
        handleFrameLoaded,
        handlePaymentError,
        handlePaymentStart,
        handlePaymentSuccess,
        paymentUrl,
    ]);

    return (
        <>
            <PaymentLayout
                footer={footer}
                orderInfo={orderInfo}
                loading={ownLoader.isVisible}
                spinnerText={ownLoader.text}
            >
                {iframe}
            </PaymentLayout>

            <ErrorModal order={order} error={orderError} />
        </>
    );
};

export default withReducers([[ELoadableReducer.BUSES_BOOK, bookReducer]])(
    memo(BookPaymentPage),
);
