import {FC, memo, useCallback, useEffect, useMemo} from 'react';
import {useDispatch} from 'react-redux';

import {IGenericOrderInfo} from 'server/api/GenericOrderApi/types/common/IGenericOrderInfo';
import EGenericOrderState from 'server/api/GenericOrderApi/types/common/EGenericOrderState';
import ECancellationReason from 'server/api/GenericOrderApi/types/common/ECancellationReason';
import EOriginalPaymentErrorCode from 'types/common/EOriginalPaymentErrorCode';
import {IBusesBookPaymentQuery} from 'types/buses/booking/IBusesBookPaymentQuery';
import {EBusesOrderError} from 'reducers/buses/book/order/types';
import {EBusesGoal} from 'utilities/metrika/types/goals/buses';

import startBookFlowAction from 'reducers/buses/book/thunk/startBookFlowAction/startBookFlowAction';
import hidePaymentErrorAndStartPaymentAction from 'reducers/buses/book/thunk/hidePaymentErrorAndStartPaymentAction';

import browserHistory from 'utilities/browserHistory/browserHistory';
import getBookUrl from 'projects/buses/utilities/urls/getBookUrl';
import getPaymentErrorTitle from 'components/BookPaymentFailedModal/utilities/getPaymentErrorTitle';
import getPaymentErrorText from 'components/BookPaymentFailedModal/utilities/getPaymentErrorText';
import useQuery from 'utilities/hooks/useQuery';
import getAction from 'projects/buses/pages/booking/BookPaymentPage/components/ErrorModal/utilities/getErrorAction';
import {reachGoal} from 'utilities/metrika';
import getSearchUrl from 'projects/buses/utilities/urls/getSearchUrl';

import * as i18n from 'i18n/trains-error-page-order';
import * as i18nBookErrors from 'i18n/buses-bookErrors';

import CommonErrorModal, {
    IErrorModalProps as ICommonErrorModalProps,
} from 'components/ErrorModal/ErrorModal';

interface IErrorModalProps {
    order: IGenericOrderInfo | null;
    error: EBusesOrderError | null;
}

const ErrorModal: FC<IErrorModalProps> = props => {
    const {order, error} = props;

    const {orderId, from, to, when, rideId, placesCount, places} =
        useQuery<keyof IBusesBookPaymentQuery>();

    const dispatch = useDispatch();

    const handleRestartPayment = useCallback(async () => {
        if (!orderId) {
            return;
        }

        await dispatch(hidePaymentErrorAndStartPaymentAction({orderId}));
        await dispatch(startBookFlowAction({orderId}));
    }, [dispatch, orderId]);

    const goToBookPage = useCallback(() => {
        if (!from || !to || !when || !rideId || !placesCount) {
            throw new Error(
                'Нет необходимых параметров для построения ссылки на страницу бронирования',
            );
        }

        browserHistory?.push(
            getBookUrl({
                from,
                to,
                when,
                rideId,
                placesCount,
                places: places || undefined,
            }),
        );
    }, [from, places, placesCount, rideId, to, when]);

    const goToSearch = useCallback(() => {
        if (!from || !to || !when) {
            throw new Error('Нет необходимых параметров');
        }

        browserHistory?.push(
            getSearchUrl({
                fromSlug: from,
                toSlug: to,
                when,
            }),
        );
    }, [from, to, when]);

    const errorInfo = useMemo((): Pick<
        ICommonErrorModalProps,
        'title' | 'text' | 'primaryAction' | 'secondaryAction'
    > | null => {
        if (!placesCount) {
            return null;
        }

        if (error) {
            return {
                title: i18nBookErrors.cantBookTickets({
                    ticketsCount: Number(placesCount),
                }),
                primaryAction: getAction(
                    i18nBookErrors.tryChooseAnotherRide(),
                    goToSearch,
                ),
            };
        }

        if (!order) {
            return null;
        }

        if (order.state === EGenericOrderState.CANCELLED) {
            if (order.cancellationReason === ECancellationReason.EXPIRED) {
                return {
                    title: i18n.paymentDashErrorDashTimeoutDashTitle(),
                    text: i18n.paymentDashErrorDashTimeoutDashContent(),
                    primaryAction: getAction(
                        i18n.buttonDashToDashPassengersDashStep(),
                        goToBookPage,
                    ),
                };
            }

            return {
                title: i18nBookErrors.cantBookTickets({
                    ticketsCount: Number(placesCount),
                }),
                primaryAction: getAction(
                    i18nBookErrors.tryChooseAnotherRide(),
                    goToSearch,
                ),
            };
        }

        const errorCode = order.payment?.errorInfo;

        if (errorCode) {
            /**
             * Происходит при перезагрузке страницы, скипаем такую ошибку
             */
            if (errorCode === EOriginalPaymentErrorCode.USER_CANCELLED) {
                return null;
            }

            return {
                title: getPaymentErrorTitle(errorCode),
                text: getPaymentErrorText(errorCode),
                primaryAction: getAction(
                    i18n.buttonDashToDashResetDashPayment(),
                    handleRestartPayment,
                ),
                secondaryAction: getAction(
                    i18n.buttonDashToDashPassengersDashStep(),
                    goToBookPage,
                ),
            };
        }

        return null;
    }, [
        error,
        goToBookPage,
        goToSearch,
        handleRestartPayment,
        order,
        placesCount,
    ]);

    useEffect(() => {
        if (errorInfo) {
            reachGoal(EBusesGoal.PAYMENT_PAGE_ERROR);
        }
    }, [errorInfo]);

    if (!errorInfo) {
        return null;
    }

    return <CommonErrorModal isVisible {...errorInfo} />;
};

export default memo(ErrorModal);
