import {memo, FC, useMemo} from 'react';
import moment from 'moment';

import {EProjectName} from 'constants/common';

import {IBusesServiceInfo} from 'server/api/GenericOrderApi/types/common/service/IBusesServiceInfo/IBusesServiceInfo';
import {isNotUndefined} from 'types/utilities';

import {HUMAN_DATETIME_WITHOUT_YEAR} from 'utilities/dateUtils/formats';
import getPointTitle from 'projects/buses/utilities/getPointTitle';
import {humanList} from 'utilities/strings/humanList';

import * as i18n from 'i18n/common';
import * as i18nPaymentPage from 'i18n/buses-PaymentPage';

import PaymentLayout from 'components/Layouts/NewPaymentLayout/PaymentLayout';

interface IOrderInfoProps {
    prettyId: string;
    serviceInfo: IBusesServiceInfo;
}

const OrderInfo: FC<IOrderInfoProps> = props => {
    const {
        prettyId,
        serviceInfo: {ride, tickets},
    } = props;

    const direction = useMemo(() => {
        const from = getPointTitle(ride.pointFrom);
        const to = getPointTitle(ride.pointTo);

        if (!from || !to) {
            return null;
        }

        return i18n.direction({
            from,
            to,
        });
    }, [ride.pointFrom, ride.pointTo]);

    const departure = useMemo(
        () => moment.utc(ride.departure).format(HUMAN_DATETIME_WITHOUT_YEAR),
        [ride.departure],
    );

    const places = useMemo(() => {
        const placesNumbers = tickets
            .map(ticket => ticket.seat)
            .filter(isNotUndefined);

        return placesNumbers.length
            ? i18nPaymentPage.places({places: humanList(placesNumbers)})
            : undefined;
    }, [tickets]);

    const additional = useMemo(
        () => [departure, places].filter(isNotUndefined).join(', '),
        [departure, places],
    );

    return (
        <PaymentLayout.PaymentOrderInfo
            orderType={EProjectName.BUSES}
            orderId={prettyId}
            descriptionGeneral={direction}
            descriptionAdditional={additional}
        />
    );
};

export default memo(OrderInfo);
