import {FC, memo, useEffect} from 'react';
import {useSelector} from 'react-redux';

import {EProjectName} from 'constants/common';

import {IBusesHappyPageQuery} from 'types/buses/booking/IBusesHappyPageQuery';
import {EOrderApiType} from 'server/api/OrdersAPI/enums/types';
import {EFooterProject} from 'components/Footer/types';
import {ELoadableReducer} from 'types/common/ELoadableReducer';
import {EBusesGoal} from 'utilities/metrika/types/goals/buses';

import happyPageReducer from 'reducers/happyPage/reducer';

import busHappyPageSelector from 'selectors/happyPage/busHappyPageSelector';

import requestHappyPageSaga from 'sagas/happyPage/orderAndCrossSales/requestHappyPageSaga';

import useQuery from 'utilities/hooks/useQuery';
import {reachGoal} from 'utilities/metrika';

import withReducers from 'containers/withReducers/withReducers';
import withSaga from 'containers/withSaga/withSaga';

import HappyPageLayout from 'components/Layouts/HappyPageLayout/HappyPageLayout';
import ErrorPage from 'projects/happyPage/components/ErrorPage/ErrorPage';
import HappyPageContent from 'projects/buses/pages/booking/HappyPage/components/HappyPageContent/HappyPageContent';

import useRequestHappyPageInfo from 'projects/happyPage/hooks/useRequestHappyPageInfo';

const HappyPage: FC = () => {
    const {orderId} = useQuery<keyof IBusesHappyPageQuery>(['orderId']);

    const happyPageInfo = useSelector(busHappyPageSelector);

    useRequestHappyPageInfo(orderId, EOrderApiType.BUS);

    useEffect(() => {
        reachGoal(EBusesGoal.HP_LOADED);
    }, []);

    if (happyPageInfo.isFailed) {
        return (
            <ErrorPage
                statusCode={happyPageInfo.errorResponse?.status}
                orderType={EProjectName.BUSES}
                footerType={EFooterProject.BUSES}
                orderId={orderId || undefined}
            />
        );
    }

    return (
        <HappyPageLayout
            orderType={EProjectName.BUSES}
            footerType={EFooterProject.BUSES}
            isLoading={happyPageInfo.isLoading}
        >
            {happyPageInfo.value && (
                <HappyPageContent
                    orderAndCrossSalesInfo={happyPageInfo.value}
                />
            )}
        </HappyPageLayout>
    );
};

export default withReducers([[ELoadableReducer.HAPPY_PAGE, happyPageReducer]])(
    withSaga([requestHappyPageSaga])(memo(HappyPage)),
);
