import React, {useEffect} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import {useParams} from 'react-router-dom';

import {EProjectName} from 'constants/common';
import {URLs} from 'constants/urls';
import {ERedirectStatusCodes} from 'constants/redirectStatusCodes';

import {ESearchFormFieldName} from 'components/SearchForm/types';
import {EBusesGoal} from 'utilities/metrika/types/goals/buses';
import {EHttpStatusCodes} from 'types/httpStatusCodes';

import initCityPageThunkAction from 'reducers/buses/city/thunk/initCityPageThunkAction';

import pointSelector from 'selectors/buses/city/pointSelector';
import nonceSelector from 'selectors/common/nonceSelector';

import requestPreviousSearchesSaga from 'sagas/index/buses/previousSearches/requestPreviousSearchesSaga';

import {deviceModMobile, deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {reachGoal} from 'utilities/metrika';
import prefetchData from './utilities/prefetchData';
import useServerDataFetcher from 'utilities/hooks/useServerDataFetcher';
import {internalUrl} from 'utilities/url';
import handleRefreshClick from 'projects/trips/components/LoadingErrorModal/utilities/handleRefreshClick';

import * as i18nBlock from 'i18n/buses-cityPage';

import withSaga from 'containers/withSaga/withSaga';

import IndexLayout from 'components/Layouts/IndexLayout/IndexLayout';
import Header from 'components/Layouts/IndexLayout/components/Header/Header';
import Footer from 'components/Layouts/IndexLayout/components/Footer/Footer';
import SearchForm from 'projects/buses/components/SearchForm/SearchForm';
import PreviousSearches from 'projects/index/IndexApp/components/IndexBuses/components/PreviousSearches/PreviousSearches';
import Heading from 'components/Heading/Heading';
import BusesAdvantages from 'projects/buses/components/BusesAdvantages/BusesAdvantages';
import IndexDirect from 'projects/index/IndexApp/components/IndexDirect/IndexDirect';
import BusesPopularDirections from 'projects/index/IndexApp/components/IndexBuses/components/PopularDirections/PopularDirections';
import Text from 'components/Text/Text';
import FooterBreadcrumbs from 'projects/buses/pages/search/components/Footer/components/FooterBreadcrumbs/FooterBreadcrumbs';
import Helmet from 'projects/buses/pages/city/CityPage/components/Helmet/Helmet';
import RedirectWithStatus from 'components/RedirectWithStatus/RedirectWithStatus';
import ErrorModal from 'components/ErrorModal/ErrorModal';
import Error500Image from 'icons/illustrations/Error500/Error500M';

import cx from './CityPage.scss';

const ROOT_QA = 'cityPage';

const CityPage: React.FC = () => {
    const {slug} = useParams<{
        slug: string;
    }>();

    const deviceType = useDeviceType();

    const nonce = useSelector(nonceSelector);
    const {value: point, error, errorStatus} = useSelector(pointSelector);
    const dispatch = useDispatch();

    const needToRender = useServerDataFetcher([prefetchData]);

    useEffect(() => {
        reachGoal(EBusesGoal.CITY_PAGE);

        dispatch(initCityPageThunkAction(slug));
    }, [dispatch, slug]);

    if (!needToRender) {
        return null;
    }

    if (error && errorStatus === EHttpStatusCodes.NotFound) {
        return (
            <RedirectWithStatus
                to={internalUrl(URLs.notFound)}
                statusCode={ERedirectStatusCodes.TEMPORARILY}
            />
        );
    }

    const from = point?.titleGenitive ?? '';

    return (
        <IndexLayout
            header={
                <Header
                    className={cx(
                        'header',
                        deviceModMobile('header', deviceType),
                    )}
                    project={EProjectName.BUSES}
                    showNavigation
                    hasSideSheetNavigation
                />
            }
            footer={<Footer project={EProjectName.BUSES} />}
        >
            {point && <Helmet city={point} />}
            <div
                className={cx('root', deviceMods('root', deviceType))}
                {...prepareQaAttributes(ROOT_QA)}
            >
                <Heading
                    className={cx('title')}
                    level={1}
                    {...prepareQaAttributes({
                        parent: ROOT_QA,
                        current: 'title',
                    })}
                >
                    {i18nBlock.title({from})}
                </Heading>

                <SearchForm autoFocusFieldName={ESearchFormFieldName.TO} />

                {deviceType.isDesktop && (
                    <PreviousSearches
                        className={cx('previousSearches')}
                        {...prepareQaAttributes({
                            parent: ROOT_QA,
                            current: 'previousSearches',
                        })}
                    />
                )}

                <BusesAdvantages
                    className={cx('advantages')}
                    {...prepareQaAttributes(ROOT_QA)}
                />

                <IndexDirect
                    className={cx('direct')}
                    projectName={EProjectName.BUSES}
                />

                <Text className={cx('description')} size="m" tag="div">
                    <p>{i18nBlock.descriptionOne({from})}</p>
                    <p>{i18nBlock.descriptionTwo()}</p>
                </Text>

                {point && <BusesPopularDirections fromId={point.key} />}

                {point && (
                    <FooterBreadcrumbs
                        className={cx('breadcrumbs')}
                        city={point}
                        nonce={nonce}
                        showDateItem={false}
                    />
                )}
            </div>
            {error && (
                <ErrorModal
                    title={i18nBlock.errorModalTitle()}
                    text={i18nBlock.errorModalText()}
                    image={<Error500Image className={cx('errorImage')} />}
                    isVisible
                    primaryAction={{
                        type: 'button',
                        title: i18nBlock.errorModalPrimaryActionTitle(),
                        handler: handleRefreshClick,
                    }}
                />
            )}
        </IndexLayout>
    );
};

export default withSaga([requestPreviousSearchesSaga])(CityPage);
