import React, {useEffect, useMemo} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import {useLocation} from 'react-router-dom';

import {EProjectName} from 'constants/common';

import {EFooterProject} from 'components/Footer/types';
import EHeaderBorderBottomType from 'components/Header/types/EHeaderBorderBottomType';
import {EBusesSearchBookingInfoStatus} from 'reducers/buses/types/search/IBusesSearchBookingInfoStore';

import {leaveSearchDatePageAction} from 'reducers/buses/search/actions';

import searchSelector from 'selectors/buses/search/searchSelector';

import getSearchQuery from 'projects/buses/utilities/search/getSearchQuery';
import useUpdateFilters from 'projects/buses/pages/search/SearchDatePage/utilities/useUpdateFilters';
import useUpdateSort from 'projects/buses/pages/search/SearchDatePage/utilities/useUpdateSort';
import useInitSearch from 'projects/buses/pages/search/SearchDatePage/utilities/useInitSearch';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import prefetchData from 'projects/buses/pages/search/SearchDatePage/utilities/prefetchData';
import useServerDataFetcher from 'utilities/hooks/useServerDataFetcher';
import {deviceMods} from 'utilities/stylesUtils';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import useMetrika from 'projects/buses/pages/search/SearchDatePage/utilities/useMetrika';
import useExpandSearchForm from 'projects/buses/pages/search/SearchDatePage/utilities/useExpandSearchForm';

import * as i18nBlock from 'i18n/buses-SearchPage';

import {TErrorActionType} from 'components/ErrorModal/ErrorModal';
import LayoutDefault from 'components/Layouts/LayoutDefault/LayoutDefault';
import SearchForm from 'projects/buses/components/SearchForm/SearchForm';
import Segments from 'projects/buses/pages/search/components/Segments/Segments';
import Title from 'projects/buses/pages/search/SearchDatePage/components/Title/Title';
import Sort from 'projects/buses/pages/search/SearchDatePage/components/Sort/Sort';
import Filters from 'projects/buses/pages/search/SearchDatePage/components/Filters/Filters';
import LayoutError500 from 'components/Layouts/LayoutError500/LayoutError500';
import Helmet from 'projects/buses/pages/search/SearchDatePage/components/Helmet/Helmet';
import Container from 'components/Layouts/Container/Container';
import Footer from 'projects/buses/pages/search/components/Footer/Footer';
import BookingInfoModalError from 'projects/buses/pages/search/components/BookingInfoModalError/BookingInfoModalError';
import SearchHeaderFormDesktopContainer from 'components/SearchHeaderFormDesktopContainer/SearchHeaderFormDesktopContainer';
import NoResultsAfterFiltration from 'projects/buses/pages/search/SearchDatePage/components/NoResultsAfterFiltration/NoResultsAfterFiltration';
import Flex from 'components/Flex/Flex';
import EmptySearch from 'projects/buses/pages/search/components/EmptySearch/EmptySearch';

import cx from './SearchDatePage.scss';

interface ISearchDatePageProps {
    fromSlug: string;
    toSlug: string;
}

const ROOT_QA = 'busesSearchDatePage';

const errorAction: TErrorActionType = {
    type: 'button',
    title: i18nBlock.errorButton(),
    handler: () => document.location.reload(),
};

const SearchDatePage: React.FC<ISearchDatePageProps> = props => {
    const {fromSlug, toSlug} = props;

    const dispatch = useDispatch();
    const location = useLocation();
    const deviceType = useDeviceType();
    const needToRender = useServerDataFetcher([prefetchData]);

    const {isDesktop, isMobile} = deviceType;

    const {
        contextInfo,
        contextInfo: {value: context},
        segmentsInfo,
        segmentsInfo: {value: segments},
        sortInfo,
        sortInfo: {sort},
        filtersInfo,
        bookingInfo,
        calendarInfo,
        calendarInfo: {value: calendar},
        nonce,
    } = useSelector(searchSelector);

    const {
        date: originWhen = '',
        lastSearchTimeMarker,
        sortBy,
        sortDirection,
        price: priceFilterValue,
        departureTime: departureTimeFilterValue,
        arrivalTime: arrivalTimeFilterValue,
        departureStation: departureStationFilterValue,
        arrivalStation: arrivalStationFilterValue,
    } = getSearchQuery(location);

    useUpdateSort(sort, sortBy, sortDirection);

    useUpdateFilters({
        filtersInfo,
        priceFilterValue,
        departureTimeFilterValue,
        arrivalTimeFilterValue,
        departureStationFilterValue,
        arrivalStationFilterValue,
    });

    useMetrika(segmentsInfo, filtersInfo);

    const initSearch = useInitSearch(
        fromSlug,
        toSlug,
        originWhen,
        lastSearchTimeMarker,
    );

    useEffect(
        () => (): void => {
            dispatch(leaveSearchDatePageAction());
        },
        [dispatch],
    );

    const searchFormIsExpanded = useExpandSearchForm(segmentsInfo);

    const visibleSegmentsCount = useMemo(() => {
        if (!segments || !filtersInfo) {
            return 0;
        }

        return segments.length - filtersInfo.hiddenSegmentsIds.size;
    }, [filtersInfo, segments]);

    const filtersBlock = useMemo(
        () => (
            <Filters
                filters={filtersInfo?.filters || null}
                itemsCount={visibleSegmentsCount}
                sortInfo={sortInfo}
                isLoading={!segmentsInfo.isFetched}
                {...prepareQaAttributes(ROOT_QA)}
            />
        ),
        [filtersInfo, segmentsInfo.isFetched, sortInfo, visibleSegmentsCount],
    );

    const desktopFiltersAndSearchFormBarNode = useMemo(() => {
        if (deviceType.isMobile) {
            return null;
        }

        return (
            <SearchHeaderFormDesktopContainer
                searchFormNode={<SearchForm />}
                belowFormNode={filtersBlock}
            />
        );
    }, [deviceType, filtersBlock]);

    const content = useMemo(() => {
        if (
            segmentsInfo.error ||
            contextInfo.isFailed ||
            calendarInfo.error ||
            bookingInfo.value?.status ===
                EBusesSearchBookingInfoStatus.INTERVAL_SERVER_ERROR
        ) {
            return (
                <LayoutError500
                    title={i18nBlock.errorTitle()}
                    subtitle={i18nBlock.errorSubtitle()}
                    action={errorAction}
                />
            );
        }

        if (
            context &&
            calendar &&
            segmentsInfo.isFetched &&
            !segments?.length
        ) {
            return <EmptySearch context={context} calendar={calendar} />;
        }

        return (
            <Container className={cx('contentContainer')}>
                <Flex className={cx('content')} flexDirection="column">
                    <Title
                        context={context}
                        isLoading={!segmentsInfo.isFetched}
                        {...prepareQaAttributes({
                            parent: ROOT_QA,
                            current: 'title',
                        })}
                    />

                    {isDesktop && (
                        <Sort
                            className={cx('sort')}
                            platform="desktop"
                            sortInfo={sortInfo}
                            isLoading={!segmentsInfo.isFetched}
                            {...prepareQaAttributes({
                                parent: ROOT_QA,
                                current: 'sortsDesktop',
                            })}
                        />
                    )}

                    <NoResultsAfterFiltration
                        className={cx('noResultsAfterFiltration')}
                        segments={segments}
                        hiddenSegmentsIds={
                            filtersInfo?.hiddenSegmentsIds || null
                        }
                        filters={filtersInfo?.filters || null}
                    />

                    <Segments
                        items={segments}
                        hiddenItemsIds={filtersInfo?.hiddenSegmentsIds || null}
                        context={context}
                        bookingInfo={bookingInfo}
                        isLoading={!segmentsInfo.isFetched}
                        when={context?.when || undefined}
                        {...prepareQaAttributes({
                            parent: ROOT_QA,
                            current: 'segments',
                        })}
                    />
                </Flex>

                {context && <Footer context={context} nonce={nonce} />}

                <BookingInfoModalError
                    bookingInfo={bookingInfo}
                    onSearch={initSearch}
                />
            </Container>
        );
    }, [
        segmentsInfo.error,
        segmentsInfo.isFetched,
        contextInfo.isFailed,
        calendarInfo.error,
        bookingInfo,
        context,
        calendar,
        segments,
        isDesktop,
        sortInfo,
        filtersInfo?.hiddenSegmentsIds,
        filtersInfo?.filters,
        nonce,
        initSearch,
    ]);

    if (!needToRender) {
        return null;
    }

    return (
        <LayoutDefault
            className={cx(deviceMods('root', deviceType))}
            footerClassName={cx('footer')}
            isFixedNavigation
            showNavigation
            showSearchForm={isMobile}
            hasSideSheetNavigation
            headerBorderBottomType={
                isDesktop
                    ? EHeaderBorderBottomType.NONE
                    : EHeaderBorderBottomType.FULL
            }
            project={EProjectName.BUSES}
            footerType={EFooterProject.BUSES}
            filters={isMobile ? filtersBlock : undefined}
            searchFormInitialIsExpanded={searchFormIsExpanded}
            {...prepareQaAttributes(ROOT_QA)}
        >
            <>
                {context && <Helmet context={context} />}
                {desktopFiltersAndSearchFormBarNode}
                {content}
            </>
        </LayoutDefault>
    );
};

export default React.memo(SearchDatePage);
