import React, {useCallback} from 'react';

import {IWithClassName} from 'types/withClassName';
import {IBusesFilters} from 'types/buses/search/filters/IBusesFilters';
import {EBusesFilter} from 'types/buses/search/filters/EBusesFilter';
import {ETimeOfDay} from 'utilities/dateUtils/types';
import {EBusesGoal} from 'utilities/metrika/types/goals/buses';
import {IBusesSearchQuery} from 'types/buses/search/IBusesSearchQuery';

import {IBusesSearchSortSelector} from 'selectors/buses/search/searchSortSelector';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import getActiveFiltersCount from 'projects/buses/pages/search/SearchDatePage/components/Filters/utilities/getActiveFiltersCount';
import isAnyFilterAvailable from 'projects/buses/pages/search/SearchDatePage/components/Filters/utilities/isAnyFilterAvailable';
import toggleValue from 'projects/trains/pages/TrainsSearchPage/components/TrainsSearchPageFilters/utilities/toggleTrainsFilterValue';
import updateSearchQuery from 'projects/buses/utilities/search/updateSearchQuery';
import {reachGoal} from 'utilities/metrika';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import scrollTo from 'utilities/dom/scrollTo';

import FiltersMobile from 'projects/buses/pages/search/SearchDatePage/components/Filters/components/FiltersMobile/FiltersMobile';
import FiltersDesktop from 'projects/buses/pages/search/SearchDatePage/components/Filters/components/FiltersDesktop/FiltersDesktop';

interface IFiltersProps extends IWithClassName, IWithQaAttributes {
    filters: IBusesFilters | null;
    itemsCount: number;
    sortInfo: IBusesSearchSortSelector;
    isLoading: boolean;
}

const Filters: React.FC<IFiltersProps> = props => {
    const {filters, itemsCount, sortInfo, isLoading} = props;

    const deviceType = useDeviceType();

    const someFilterIsAvailable = filters
        ? isAnyFilterAvailable(filters)
        : false;

    const handleFilterChange = useCallback(
        (updatedQuery: IBusesSearchQuery, metrikaGoal?: EBusesGoal) => {
            updateSearchQuery(updatedQuery);
            scrollTo({top: 0, behavior: 'smooth'});

            if (metrikaGoal) {
                reachGoal(metrikaGoal);
            }
        },
        [],
    );

    const handlePriceFilterChange = useCallback(
        (values: [number, number]) => {
            handleFilterChange(
                {
                    price: values.map(String),
                },
                EBusesGoal.SEARCH_FILTER_PRICE_CLICK,
            );
        },
        [handleFilterChange],
    );

    const handleDepartureTimeChange = useCallback(
        (option: ETimeOfDay) => {
            if (!filters) {
                return;
            }

            handleFilterChange(
                {
                    departureTime: toggleValue(
                        filters[EBusesFilter.DEPARTURE_TIME].value,
                        option,
                    ),
                },
                EBusesGoal.SEARCH_FILTER_DEPARTURE_TIME_CLICK,
            );
        },
        [filters, handleFilterChange],
    );

    const handleArrivalTimeChange = useCallback(
        (option: ETimeOfDay) => {
            if (!filters) {
                return;
            }

            handleFilterChange(
                {
                    arrivalTime: toggleValue(
                        filters[EBusesFilter.ARRIVAL_TIME].value,
                        option,
                    ),
                },
                EBusesGoal.SEARCH_FILTER_ARRIVAL_TIME_CLICK,
            );
        },
        [filters, handleFilterChange],
    );

    const handleDepartureStationChange = useCallback(
        (option: string) => {
            if (!filters) {
                return;
            }

            handleFilterChange(
                {
                    departureStation: toggleValue(
                        filters[EBusesFilter.DEPARTURE_STATION].value,
                        option,
                    ),
                },
                EBusesGoal.SEARCH_FILTER_DEPARTURE_STATION_CLICK,
            );
        },
        [filters, handleFilterChange],
    );

    const handleArrivalStationChange = useCallback(
        (option: string) => {
            if (!filters) {
                return;
            }

            handleFilterChange(
                {
                    arrivalStation: toggleValue(
                        filters[EBusesFilter.ARRIVAL_STATION].value,
                        option,
                    ),
                },
                EBusesGoal.SEARCH_FILTER_ARRIVAL_STATION_CLICK,
            );
        },
        [filters, handleFilterChange],
    );

    const handleReset = useCallback(() => {
        handleFilterChange({
            ...(deviceType.isMobile
                ? {
                      sortBy: undefined,
                      sortDirection: undefined,
                  }
                : {}),
            sortDirection: undefined,
            price: undefined,
            departureTime: undefined,
            arrivalTime: undefined,
            departureStation: undefined,
            arrivalStation: undefined,
        });
    }, [deviceType.isMobile, handleFilterChange]);

    if (!isLoading && !someFilterIsAvailable) {
        return null;
    }

    if (deviceType.isMobile) {
        const activeFiltersCount = filters ? getActiveFiltersCount(filters) : 0;

        return (
            <FiltersMobile
                itemsCount={itemsCount}
                activeFiltersCount={activeFiltersCount}
                sortInfo={sortInfo}
                filters={filters}
                isLoading={isLoading}
                onPriceChange={handlePriceFilterChange}
                onDepartureTimeChange={handleDepartureTimeChange}
                onArrivalTimeChange={handleArrivalTimeChange}
                onDepartureStationChange={handleDepartureStationChange}
                onArrivalStationChange={handleArrivalStationChange}
                onReset={handleReset}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'filtersAndSortsMobile',
                })}
            />
        );
    }

    return (
        <FiltersDesktop
            filters={filters}
            isLoading={isLoading}
            onPriceChange={handlePriceFilterChange}
            onDepartureTimeChange={handleDepartureTimeChange}
            onArrivalTimeChange={handleArrivalTimeChange}
            onDepartureStationChange={handleDepartureStationChange}
            onArrivalStationChange={handleArrivalStationChange}
            onReset={handleReset}
            {...prepareQaAttributes({
                parent: props,
                current: 'filtersDesktop',
            })}
        />
    );
};

export default React.memo(Filters);
