import React, {useMemo} from 'react';

import {EBusesFilter} from 'types/buses/search/filters/EBusesFilter';
import {IBusesFilters} from 'types/buses/search/filters/IBusesFilters';
import {ETimeOfDay} from 'utilities/dateUtils/types';

import getActiveFiltersCount from 'projects/buses/pages/search/SearchDatePage/components/Filters/utilities/getActiveFiltersCount';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import CommonFiltersDesktop from 'components/FiltersDesktop/FiltersDesktop';
import PriceFilter from 'projects/buses/pages/search/SearchDatePage/components/Filters/components/FiltersDesktop/components/PriceFilter/PriceFilter';
import DepartureAndArrivalTimeFilter from 'projects/buses/pages/search/SearchDatePage/components/Filters/components/FiltersDesktop/components/DepartureAndArrivalTimeFilter/DepartureAndArrivalTimeFilter';
import DepartureAndArrivalStationFilter from 'projects/buses/pages/search/SearchDatePage/components/Filters/components/FiltersDesktop/components/DepartureAndArrivalStationFilter/DepartureAndArrivalStationFilter';

interface IFiltersDesktopProps extends IWithQaAttributes {
    filters: IBusesFilters | null;
    isLoading: boolean;
    onPriceChange(values: [number, number]): void;
    onDepartureTimeChange(option: ETimeOfDay): void;
    onArrivalTimeChange(option: ETimeOfDay): void;
    onDepartureStationChange(option: string): void;
    onArrivalStationChange(option: string): void;
    onReset(): void;
}

const SKELETONS_WIDTHS = [83, 242, 268];

const FiltersDesktop: React.FC<IFiltersDesktopProps> = props => {
    const {
        filters,
        isLoading,
        onPriceChange,
        onDepartureTimeChange,
        onArrivalTimeChange,
        onDepartureStationChange,
        onArrivalStationChange,
        onReset,
    } = props;

    const isAnyFilterActive = filters
        ? getActiveFiltersCount(filters) > 0
        : false;

    const content = useMemo(() => {
        if (!filters) {
            return null;
        }

        const {
            [EBusesFilter.PRICE]: {range: priceRange, value: priceValue},
            [EBusesFilter.DEPARTURE_TIME]: {
                options: departureTimeOptions,
                value: departureTimeValue,
                availableOptions: departureTimeAvailableOptions,
            },
            [EBusesFilter.ARRIVAL_TIME]: {
                options: arrivalTimeOptions,
                value: arrivalTimeValue,
                availableOptions: arrivalTimeAvailableOptions,
            },
            [EBusesFilter.DEPARTURE_STATION]: {
                options: departureStationOptions,
                value: departureStationValue,
                availableOptions: departureStationAvailableOptions,
            },
            [EBusesFilter.ARRIVAL_STATION]: {
                options: arrivalStationOptions,
                value: arrivalStationValue,
                availableOptions: arrivalStationAvailableOptions,
            },
        } = filters;

        return (
            <>
                {priceRange && (
                    <PriceFilter
                        values={priceValue || priceRange}
                        rangeValues={priceRange}
                        onChange={onPriceChange}
                    />
                )}

                <DepartureAndArrivalTimeFilter
                    departureValue={departureTimeValue}
                    departureOptions={departureTimeOptions}
                    departureAvailableOptions={departureTimeAvailableOptions}
                    onDepartureChange={onDepartureTimeChange}
                    arrivalValue={arrivalTimeValue}
                    arrivalOptions={arrivalTimeOptions}
                    arrivalAvailableOptions={arrivalTimeAvailableOptions}
                    onArrivalChange={onArrivalTimeChange}
                />

                <DepartureAndArrivalStationFilter
                    departureValue={departureStationValue}
                    departureOptions={departureStationOptions}
                    departureAvailableOptions={departureStationAvailableOptions}
                    onDepartureChange={onDepartureStationChange}
                    arrivalValue={arrivalStationValue}
                    arrivalOptions={arrivalStationOptions}
                    arrivalAvailableOptions={arrivalStationAvailableOptions}
                    onArrivalChange={onArrivalStationChange}
                />
            </>
        );
    }, [
        filters,
        onPriceChange,
        onDepartureTimeChange,
        onArrivalTimeChange,
        onDepartureStationChange,
        onArrivalStationChange,
    ]);

    return (
        <CommonFiltersDesktop
            canReset={isAnyFilterActive}
            onReset={onReset}
            loading={isLoading}
            skeletonsWidth={SKELETONS_WIDTHS}
            withShadow={false}
            withoutHorizontalPaddings
            {...prepareQaAttributes(props)}
        >
            {content}
        </CommonFiltersDesktop>
    );
};

export default React.memo(FiltersDesktop);
