import React from 'react';

import {IBusesFilters} from 'types/buses/search/filters/IBusesFilters';
import {ETimeOfDay} from 'utilities/dateUtils/types';
import {EBusesFilter} from 'types/buses/search/filters/EBusesFilter';

import {IBusesSearchSortSelector} from 'selectors/buses/search/searchSortSelector';

import {useDeviceType} from 'utilities/hooks/useDeviceType';

import * as i18nCommonFiltersBlock from 'i18n/common-filters';

import Separator from 'components/Separator/Separator';
import Raskrivashka from 'components/Raskrivashka/Raskrivashka';
import PriceFilter from 'components/PriceFilter/PriceFilter';
import TimeFilter from 'components/TimeFilter/TimeFilter';
import FilterCheckList from 'components/FilterCheckList/FilterCheckList';
import Intersperse from 'components/Intersperse/Intersperse';
import Sort from 'projects/buses/pages/search/SearchDatePage/components/Sort/Sort';

import cx from './ModalContent.scss';

interface IModalContentProps {
    sortInfo: IBusesSearchSortSelector;
    filters: IBusesFilters;
    onPriceChange(values: [number, number]): void;
    onDepartureTimeChange(option: ETimeOfDay): void;
    onArrivalTimeChange(option: ETimeOfDay): void;
    onDepartureStationChange(option: string): void;
    onArrivalStationChange(option: string): void;
}

const ModalContent: React.FC<IModalContentProps> = props => {
    const {
        filters: {
            [EBusesFilter.PRICE]: priceFilter,
            [EBusesFilter.DEPARTURE_TIME]: departureTimeFilter,
            [EBusesFilter.ARRIVAL_TIME]: arrivalTimeFilter,
            [EBusesFilter.DEPARTURE_STATION]: departureStationFilter,
            [EBusesFilter.ARRIVAL_STATION]: arrivalStationFilter,
        },
        sortInfo,
        onPriceChange,
        onDepartureTimeChange,
        onArrivalTimeChange,
        onDepartureStationChange,
        onArrivalStationChange,
    } = props;

    const deviceType = useDeviceType();

    return (
        <Intersperse separator={<Separator />}>
            {priceFilter.range && (
                <Raskrivashka
                    contentIsInvolved={Boolean(
                        priceFilter.value &&
                            (priceFilter.value[0] !== priceFilter.range[0] ||
                                priceFilter.value[1] !== priceFilter.range[1]),
                    )}
                    label={i18nCommonFiltersBlock.price()}
                    withPrerender={false}
                >
                    <PriceFilter
                        className={cx('modalItem', cx('price'))}
                        values={priceFilter.value || priceFilter.range}
                        rangeValues={priceFilter.range}
                        onChange={onPriceChange}
                    />
                </Raskrivashka>
            )}

            {departureTimeFilter.options.length > 1 && (
                <Raskrivashka
                    contentIsInvolved={Boolean(
                        departureTimeFilter.value.length,
                    )}
                    label={i18nCommonFiltersBlock.departureTime()}
                    withPrerender={false}
                >
                    <TimeFilter
                        className={cx('modalItem')}
                        deviceType={deviceType}
                        value={departureTimeFilter.value}
                        options={departureTimeFilter.options}
                        availableValues={departureTimeFilter.availableOptions}
                        onChange={onDepartureTimeChange}
                    />
                </Raskrivashka>
            )}

            {arrivalTimeFilter.options.length > 1 && (
                <Raskrivashka
                    contentIsInvolved={Boolean(arrivalTimeFilter.value.length)}
                    label={i18nCommonFiltersBlock.arrivalTime()}
                    withPrerender={false}
                >
                    <TimeFilter
                        className={cx('modalItem')}
                        deviceType={deviceType}
                        value={arrivalTimeFilter.value}
                        options={arrivalTimeFilter.options}
                        availableValues={arrivalTimeFilter.availableOptions}
                        onChange={onArrivalTimeChange}
                    />
                </Raskrivashka>
            )}

            {departureStationFilter.options.length > 1 && (
                <Raskrivashka
                    contentIsInvolved={Boolean(
                        departureStationFilter.value.length,
                    )}
                    label={i18nCommonFiltersBlock.departureStation()}
                    withPrerender={false}
                >
                    <FilterCheckList
                        className={cx('modalItem')}
                        deviceType={deviceType}
                        values={departureStationFilter.value}
                        options={departureStationFilter.options}
                        availableValues={
                            departureStationFilter.availableOptions
                        }
                        onChange={onDepartureStationChange}
                    />
                </Raskrivashka>
            )}

            {arrivalStationFilter.options.length > 1 && (
                <Raskrivashka
                    contentIsInvolved={Boolean(
                        arrivalStationFilter.value.length,
                    )}
                    label={i18nCommonFiltersBlock.arrivalStation()}
                    withPrerender={false}
                >
                    <FilterCheckList
                        className={cx('modalItem')}
                        deviceType={deviceType}
                        values={arrivalStationFilter.value}
                        options={arrivalStationFilter.options}
                        availableValues={arrivalStationFilter.availableOptions}
                        onChange={onArrivalStationChange}
                    />
                </Raskrivashka>
            )}

            <Sort
                className={cx('modalItem')}
                platform="mobile"
                sortInfo={sortInfo}
                above={5}
                below={8}
            />
        </Intersperse>
    );
};

export default React.memo(ModalContent);
