import React from 'react';

import {IBusesSegment} from 'types/buses/search/IBusesSegment';
import {IBusesFilters} from 'types/buses/search/filters/IBusesFilters';
import {EBusesFilter} from 'types/buses/search/filters/EBusesFilter';
import {IWithClassName} from 'types/withClassName';

import updateSearchQuery from 'projects/buses/utilities/search/updateSearchQuery';

import * as i18nBlock from 'i18n/common-filters';

import CommonNoResultsAfterFiltration, {
    IFilterBadgeProps,
} from 'components/NoResultsAfterFiltration/NoResultsAfterFiltration';
import CardWithDeviceLayout from 'components/CardWithDeviceLayout/CardWithDeviceLayout';

import cx from './NoResultsAfterFiltration.scss';

interface INoResultsAfterFiltrationProps extends IWithClassName {
    segments: IBusesSegment[] | null;
    hiddenSegmentsIds: Set<string> | null;
    filters: IBusesFilters | null;
}

const FILTERS: EBusesFilter[] = [
    EBusesFilter.PRICE,
    EBusesFilter.DEPARTURE_TIME,
    EBusesFilter.ARRIVAL_TIME,
    EBusesFilter.DEPARTURE_STATION,
    EBusesFilter.ARRIVAL_STATION,
];

const NoResultsAfterFiltration: React.FC<INoResultsAfterFiltrationProps> =
    props => {
        const {className, segments, hiddenSegmentsIds, filters} = props;

        if (
            !segments?.length ||
            !hiddenSegmentsIds?.size ||
            !filters ||
            segments.length !== hiddenSegmentsIds.size
        ) {
            return null;
        }

        const badgeFilters: IFilterBadgeProps[] = [];

        FILTERS.forEach(type => {
            const filter = filters[type];

            if (filter.hiddenByFilter.size) {
                badgeFilters.push({
                    text: i18nBlock[type](),
                    handler: () => {
                        updateSearchQuery({[type]: undefined});
                    },
                });
            }
        });

        return (
            <CardWithDeviceLayout className={cx('root', className)}>
                <CommonNoResultsAfterFiltration
                    className={cx('wrapper')}
                    filters={badgeFilters}
                />
            </CardWithDeviceLayout>
        );
    };

export default React.memo(NoResultsAfterFiltration);
