import React, {useCallback} from 'react';

import {
    EBusesSortDirection,
    EBusesSortType,
} from 'types/buses/search/EBusesSortType';
import {ESortType} from 'types/common/sort/ESortType';
import {ESortDirection} from 'types/common/sort/ESortDirection';
import {IWithClassName} from 'types/withClassName';
import {EBusesGoal} from 'utilities/metrika/types/goals/buses';

import {IBusesSearchSortSelector} from 'selectors/buses/search/searchSortSelector';

import updateSearchQuery from 'projects/buses/utilities/search/updateSearchQuery';
import {reachGoal} from 'utilities/metrika';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import SortDesktop from 'components/SortDesktop/SortDesktop';
import SortMobile from 'components/SortMobile/SortMobile';
import {IBoxProps} from 'components/Box/Box';

interface ICommonSortingProps extends IWithClassName {
    sortInfo: IBusesSearchSortSelector;
}

interface IDesktopSortingProps extends ICommonSortingProps, IWithQaAttributes {
    platform: 'desktop';
    isLoading: boolean;
}

interface IMobileSortingProps
    extends ICommonSortingProps,
        Partial<Pick<IBoxProps, 'above' | 'below'>> {
    platform: 'mobile';
}

type TSortingProps = IDesktopSortingProps | IMobileSortingProps;

const Sort: React.FC<TSortingProps> = props => {
    const {
        className,
        sortInfo: {sort, availableSorts},
    } = props;

    const handleChange = useCallback(
        (sortType: EBusesSortType, sortDirection: EBusesSortDirection) => {
            if (sortType !== sort.type) {
                reachGoal(EBusesGoal.SEARCH_SORT_TYPE_CLICK);
            }

            if (sortDirection !== sort.direction) {
                reachGoal(EBusesGoal.SEARCH_SORT_DIRECTION_CLICK);
            }

            const isInitialSort =
                sortType === ESortType.DEPARTURE &&
                sortDirection === ESortDirection.ASC;

            updateSearchQuery({
                sortBy: isInitialSort ? undefined : sortType,
                sortDirection: isInitialSort ? undefined : sortDirection,
            });
        },
        [sort],
    );

    if (props.platform === 'mobile') {
        if (!availableSorts.length) {
            return null;
        }

        return (
            <SortMobile
                className={className}
                type={sort.type}
                direction={sort.direction}
                sortOrder={availableSorts}
                above={props.above}
                below={props.below}
                onChange={handleChange}
            />
        );
    }

    if (!availableSorts.length && !props.isLoading) {
        return null;
    }

    return (
        <SortDesktop
            className={className}
            type={sort.type}
            direction={sort.direction}
            sortOrder={availableSorts}
            loading={props.isLoading}
            onClick={handleChange}
            {...prepareQaAttributes(props)}
        />
    );
};

export default React.memo(Sort);
