import React, {useMemo} from 'react';

import {IBusesFilledContext} from 'types/buses/common/IBusesContext';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import getHumanWhen from 'utilities/dateUtils/when/getHumanWhen';

import * as i18nBlock from 'i18n/buses-SearchPage';

import Box from 'components/Box/Box';
import TextSkeleton from 'components/Skeletons/TextSkeleton/TextSkeleton';

import cx from './Title.scss';

interface ITitleProps extends IWithQaAttributes {
    context: IBusesFilledContext | null;
    isLoading: boolean;
}

const Title: React.FC<ITitleProps> = props => {
    const {context, isLoading} = props;

    const deviceType = useDeviceType();

    const content = useMemo(() => {
        if (isLoading || !context) {
            if (deviceType.isMobile) {
                return (
                    <>
                        <TextSkeleton size="l" withAnimation />

                        <TextSkeleton
                            className={cx('skeleton')}
                            size="l"
                            withAnimation
                        />
                    </>
                );
            }

            return (
                <TextSkeleton
                    className={cx('skeleton')}
                    size="xl"
                    withAnimation
                />
            );
        }

        const {from, to, originWhen} = context;

        if (!originWhen) {
            return null;
        }

        return i18nBlock.searchTitle({
            from: from.titleGenitive || from.title,
            to: to.titleAccusative || to.title,
            date: getHumanWhen(originWhen),
        });
    }, [isLoading, context, deviceType.isMobile]);

    return (
        <Box
            className={cx('root', deviceMods('root', deviceType))}
            above={deviceType.isMobile ? 7 : 10}
            textSize={deviceType.isMobile ? 'l' : 'xl'}
            textWeight="bold"
            {...prepareQaAttributes(props)}
        >
            {content}
        </Box>
    );
};

export default React.memo(Title);
