import {useEffect} from 'react';
import isEqual from 'lodash/isEqual';
import {useDispatch} from 'react-redux';

import {IBusesSearchFiltersStore} from 'reducers/buses/types/search/IBusesSearchFiltersStore';
import {EBusesFilter} from 'types/buses/search/filters/EBusesFilter';
import {ETimeOfDay} from 'utilities/dateUtils/types';

import {updateFiltersAction} from 'reducers/buses/search/filters/actions';

import {IBusesSearchFiltersSelector} from 'selectors/buses/search/searchFiltersSelector';

import getFormattedFilterValue from 'projects/buses/utilities/search/getFormattedFilterValue';

export default function useUpdateFilters({
    filtersInfo,
    priceFilterValue,
    departureTimeFilterValue,
    arrivalTimeFilterValue,
    departureStationFilterValue,
    arrivalStationFilterValue,
}: {
    filtersInfo: IBusesSearchFiltersSelector | null;
    priceFilterValue: string[] | undefined;
    departureTimeFilterValue: ETimeOfDay | ETimeOfDay[] | undefined;
    arrivalTimeFilterValue: ETimeOfDay | ETimeOfDay[] | undefined;
    departureStationFilterValue: string | string[] | undefined;
    arrivalStationFilterValue: string | string[] | undefined;
}): void {
    const dispatch = useDispatch();

    useEffect(() => {
        if (!filtersInfo) {
            return;
        }

        const {filters} = filtersInfo;

        const filtersToUpdate = [
            {type: EBusesFilter.PRICE, value: priceFilterValue},
            {
                type: EBusesFilter.DEPARTURE_TIME,
                value: departureTimeFilterValue,
            },
            {type: EBusesFilter.ARRIVAL_TIME, value: arrivalTimeFilterValue},
            {
                type: EBusesFilter.DEPARTURE_STATION,
                value: departureStationFilterValue,
            },
            {
                type: EBusesFilter.ARRIVAL_STATION,
                value: arrivalStationFilterValue,
            },
        ].reduce<Partial<IBusesSearchFiltersStore>>(
            (accFiltersToUpdate, typeAndValue) => {
                const filterValue = filters[typeAndValue.type].value;

                const formattedValue =
                    typeAndValue.type === EBusesFilter.PRICE
                        ? typeAndValue.value
                            ? [
                                  Number(typeAndValue.value[0]),
                                  Number(typeAndValue.value[1]),
                              ]
                            : null
                        : getFormattedFilterValue(typeAndValue.value || []);

                if (!isEqual(filterValue, formattedValue)) {
                    accFiltersToUpdate[typeAndValue.type] =
                        formattedValue as any;
                }

                return accFiltersToUpdate;
            },
            {},
        );

        if (Object.keys(filtersToUpdate).length) {
            dispatch(updateFiltersAction(filtersToUpdate));
        }
    }, [
        dispatch,
        filtersInfo,
        priceFilterValue,
        departureTimeFilterValue,
        arrivalTimeFilterValue,
        departureStationFilterValue,
        arrivalStationFilterValue,
    ]);
}
