import {FC, Fragment, useEffect, useMemo, memo} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import {EProjectName} from 'constants/common';
import {EBusesDirectionBlock} from 'projects/buses/constants/direction/EBusesDirectionBlock';

import EAsyncStatus from 'types/common/EAsyncStatus';
import {EFooterProject} from 'components/Footer/types';
import {ELoadableReducer} from 'types/common/ELoadableReducer';
import {EBusesGoal} from 'utilities/metrika/types/goals/buses';
import {EBusesSearchBookingInfoStatus} from 'reducers/buses/types/search/IBusesSearchBookingInfoStore';
import {ECommonLandingBlockType} from 'types/common/seoPages/ECommonLandingBlockType';
import {IBusesSegmentsBlock} from 'types/buses/direction/IBusesSegmentsBlock';

import directionPageReducer from 'reducers/buses/directionPage/reducer';
import {fetchDirectionPageInfo} from 'reducers/buses/directionPage/thunk/fetchDirectionPageInfo';

import searchDirectionSelector from 'selectors/buses/search/searchDirectionSelector';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import useServerDataFetcher from 'utilities/hooks/useServerDataFetcher';
import prefetchData from 'projects/buses/pages/search/SearchDirectionPage/utilities/prefetchData';
import {reachGoal} from 'utilities/metrika';
import {useReachGoal} from 'utilities/metrika/useReachGoal';

import * as i18nBlock from 'i18n/buses-SearchPage';

import withReducers from 'containers/withReducers/withReducers';

import Container from 'components/Layouts/Container/Container';
import {TErrorActionType} from 'components/ErrorModal/ErrorModal';
import LayoutLight from 'components/Layouts/LayoutLight/LayoutLight';
import SearchForm from 'projects/buses/components/SearchForm/SearchForm';
import Footer from 'projects/buses/pages/search/components/Footer/Footer';
import LayoutError500 from 'components/Layouts/LayoutError500/LayoutError500';
import SegmentsSkeleton from './components/SegmentsSkeleton/SegmentsSkeleton';
import CrossSaleHotelsBlock from 'components/CrossSaleHotelsBlock/CrossSaleHotelsBlock';
import EmptySearch from 'projects/buses/pages/search/components/EmptySearch/EmptySearch';
import Helmet from 'projects/buses/pages/search/SearchDirectionPage/components/Helmet/Helmet';
import CrossLinks from 'projects/buses/pages/search/SearchDirectionPage/components/CrossLinks/CrossLinks';
import DirectionInfo from 'projects/buses/pages/search/SearchDirectionPage/components/DirectionInfo/DirectionInfo';
import ClosestDateSegments from 'projects/buses/pages/search/SearchDirectionPage/components/ClosestDateSegments/ClosestDateSegments';
import TitleWithDescription from 'projects/buses/pages/search/SearchDirectionPage/components/TitleWithDescription/TitleWithDescription';

import cx from './SearchDirectionPage.scss';

interface ISearchDirectionPageProps {
    fromSlug: string;
    toSlug: string;
}

const errorAction: TErrorActionType = {
    type: 'button',
    title: i18nBlock.errorButton(),
    handler: () => document.location.reload(),
};

const SearchDirectionPage: FC<ISearchDirectionPageProps> = props => {
    const {fromSlug, toSlug} = props;

    const needToRender = useServerDataFetcher([prefetchData(fromSlug, toSlug)]);
    const deviceType = useDeviceType();
    const dispatch = useDispatch();

    const {
        directionData: {status, data},
        searchStats,
        bookingInfo,
        nonce,
    } = useSelector(searchDirectionSelector);

    const {context, blocks} = data || {};
    const {minPrice, maxPrice} = searchStats || {};

    const isLoading =
        status === EAsyncStatus.INITIAL || status === EAsyncStatus.LOADING;

    useEffect(() => {
        reachGoal(EBusesGoal.SEARCH_DIRECTION);

        const shouldFetchDirection =
            status === EAsyncStatus.INITIAL ||
            context?.from.slug !== fromSlug ||
            context?.to.slug !== toSlug;

        if (status !== EAsyncStatus.ERROR && shouldFetchDirection) {
            dispatch(fetchDirectionPageInfo(fromSlug, toSlug));
        }
    }, [
        status,
        dispatch,
        fromSlug,
        toSlug,
        context?.from.slug,
        context?.to.slug,
    ]);

    const handleCalendarDateClick = useReachGoal(
        EBusesGoal.SEARCH_DIRECTION_CALENDAR_DATE_CLICK,
    );

    const segments = useMemo(() => {
        return data?.blocks.find(
            (block): block is IBusesSegmentsBlock =>
                block.type === EBusesDirectionBlock.BUSES_SEGMENTS,
        )?.data.segments;
    }, [data?.blocks]);

    const desktopForm = useMemo(() => {
        if (deviceType.isMobile) {
            return null;
        }

        return (
            <div className={cx('formWrapper')}>
                <Container>
                    <SearchForm
                        initiallyCalendarIsOpen
                        onDateClick={handleCalendarDateClick}
                    />
                </Container>
            </div>
        );
    }, [deviceType, handleCalendarDateClick]);

    const mobileForm = useMemo(() => {
        if (deviceType.isDesktop) {
            return null;
        }

        return <SearchForm className={cx('form')} />;
    }, [deviceType]);

    const content = useMemo(() => {
        if (!needToRender) {
            return null;
        }

        if (
            status === EAsyncStatus.ERROR ||
            context?.errors.length ||
            bookingInfo.value?.status ===
                EBusesSearchBookingInfoStatus.INTERVAL_SERVER_ERROR
        ) {
            return (
                <LayoutError500
                    title={i18nBlock.errorTitle()}
                    subtitle={i18nBlock.errorSubtitle()}
                    action={errorAction}
                />
            );
        }

        if (context && !isLoading && !segments?.length) {
            const form = desktopForm || <Container>{mobileForm}</Container>;

            return (
                <>
                    {form}

                    <EmptySearch context={context} />
                </>
            );
        }

        return (
            <>
                {desktopForm}

                <Container className={cx('contentContainer')}>
                    <div className={cx('content')}>
                        <TitleWithDescription
                            context={context}
                            isLoading={isLoading}
                        />

                        {mobileForm}

                        {isLoading && (
                            <SegmentsSkeleton className={cx('segments')} />
                        )}

                        {blocks &&
                            blocks.map((block, index) => {
                                switch (block.type) {
                                    case EBusesDirectionBlock.BUSES_SEGMENTS:
                                        return (
                                            <ClosestDateSegments
                                                className={cx('segments')}
                                                key={index}
                                                context={context}
                                                block={block.data}
                                                bookingInfo={bookingInfo}
                                            />
                                        );

                                    case EBusesDirectionBlock.BUSES_CROSSLINK:
                                        return (
                                            <Fragment key={index}>
                                                {segments && context && (
                                                    <DirectionInfo
                                                        className={cx(
                                                            'directionInfo',
                                                        )}
                                                        context={context}
                                                        segments={segments}
                                                        minPrice={minPrice}
                                                        maxPrice={maxPrice}
                                                    />
                                                )}

                                                <CrossLinks
                                                    className={cx('crossLinks')}
                                                    crossLinks={block.data}
                                                />
                                            </Fragment>
                                        );

                                    case ECommonLandingBlockType.HOTELS_CROSS_SALE:
                                        return (
                                            <CrossSaleHotelsBlock
                                                {...block.data}
                                                key={`${block.type}${index}`}
                                                vertical={EProjectName.BUSES}
                                                className={cx(
                                                    'hotelsCrossSaleBlock',
                                                )}
                                                linkType="region"
                                            />
                                        );

                                    default:
                                        return null;
                                }
                            })}
                    </div>

                    {context && <Footer context={context} nonce={nonce} />}
                </Container>
            </>
        );
    }, [
        needToRender,
        bookingInfo,
        context,
        segments,
        desktopForm,
        mobileForm,
        minPrice,
        maxPrice,
        nonce,
        blocks,
        isLoading,
        status,
    ]);

    if (!needToRender) {
        return null;
    }

    return (
        <LayoutLight
            className={cx(deviceMods('root', deviceType))}
            headerClassName={cx('header')}
            footerClassName={cx('footer')}
            project={EProjectName.BUSES}
            footerType={EFooterProject.BUSES}
        >
            {context && (
                <Helmet
                    context={context}
                    minPrice={minPrice || null}
                    nonce={nonce}
                />
            )}

            {content}
        </LayoutLight>
    );
};

export default withReducers<ISearchDirectionPageProps>([
    [ELoadableReducer.BUSES_DIRECTION_PAGE, directionPageReducer],
])(memo(SearchDirectionPage));
