import React, {useMemo} from 'react';

import {IWithClassName} from 'types/withClassName';
import {IBusesSegment} from 'types/buses/search/IBusesSegment';
import {IBusesFilledContext} from 'types/buses/common/IBusesContext';
import {IBusesSegmentsInfo} from 'types/buses/direction/IBusesSegmentsInfo';
import {IBusesSearchBookingInfoStore} from 'reducers/buses/types/search/IBusesSearchBookingInfoStore';

import {useBoolean} from 'utilities/hooks/useBoolean';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import getHumanWhen from 'utilities/dateUtils/when/getHumanWhen';

import * as i18nBlock from 'i18n/buses-SearchDirectionPage';

import Box from 'components/Box/Box';
import Button from 'components/Button/Button';
import Heading from 'components/Heading/Heading';
import Segments from 'projects/buses/pages/search/components/Segments/Segments';

interface IClosestDateSegmentsProps extends IWithClassName {
    context?: IBusesFilledContext;
    block: IBusesSegmentsInfo;
    bookingInfo: IBusesSearchBookingInfoStore;
}

const VISIBLE_SEGMENTS_COUNT = 5;
const ADFOX_POSITION = 2;

const ClosestDateSegments: React.FC<IClosestDateSegmentsProps> = props => {
    const {className, block, context, bookingInfo} = props;
    const {segments, when} = block;

    const {value: isOpen, setTrue: open} = useBoolean(false);

    const deviceType = useDeviceType();

    const preparedSegments: IBusesSegment[] = useMemo(() => {
        return segments.map(segment => ({
            ...segment,
            from: {
                ...segment.from,
                description: segment.from.fullTitle,
            },
            to: {
                ...segment.to,
                description: segment.to.fullTitle,
            },
            fromDesc: segment.from.title,
            toDesc: segment.to.title,
            arrivalTime: segment.arrivalTime ? segment.arrivalTime : null,
            departureTime: segment.departureTime,
            duration: segment.duration || null,
            price: {
                amount: segment.price.value,
            },
        }));
    }, [segments]);

    const title = useMemo(() => {
        return (
            context &&
            when && (
                <Heading level={2} size={deviceType.isDesktop ? 'xl' : 'l'}>
                    {i18nBlock.segmentsTitleWithDate({
                        from: context.from.popularTitle || context.from.title,
                        to: context.to.popularTitle || context.to.title,
                        date: getHumanWhen(when, {
                            timezone: context.from.timezone,
                        }),
                    })}
                </Heading>
            )
        );
    }, [context, when, deviceType.isDesktop]);

    return (
        <Box className={className} between={5}>
            {title}

            <Segments
                items={
                    (isOpen
                        ? preparedSegments
                        : preparedSegments?.slice(0, VISIBLE_SEGMENTS_COUNT)) ||
                    null
                }
                context={context}
                bookingInfo={bookingInfo}
                advertisingPosition={ADFOX_POSITION}
                when={block.when}
            />

            {!isOpen && segments?.length > VISIBLE_SEGMENTS_COUNT && (
                <Button
                    width={deviceType.isMobile ? 'max' : 'auto'}
                    onClick={open}
                >
                    {i18nBlock.showAllSegmentsButton()}
                </Button>
            )}
        </Box>
    );
};

export default React.memo(ClosestDateSegments);
