import React from 'react';
import moment from 'moment-timezone';
import head from 'lodash/head';
import last from 'lodash/last';

import {IWithClassName} from 'types/withClassName';
import {IBusesFilledContext} from 'types/buses/common/IBusesContext';
import {IBusesSeoSegment} from 'types/buses/direction/IBusesSeoSegment';

import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import {CurrencyType} from 'utilities/currency/CurrencyType';
import {formatDate, parseDate} from 'utilities/dateUtils';
import {ROBOT_DATETIME, TIME} from 'utilities/dateUtils/formats';
import {getDuration} from 'projects/avia/lib/date/getDuration';
import formatInteger from 'utilities/numbers/formatInteger';

import * as i18nCommonUnitsBlock from 'i18n/common-units';
import * as i18nDurationBlock from 'i18n/avia-Duration';
import * as i18nBlock from 'i18n/buses-SearchDirectionPage';

import Box from 'components/Box/Box';
import Heading from 'components/Heading/Heading';
import Text from 'components/Text/Text';
import Price from 'components/Price/Price';

import cx from './DirectionInfo.scss';

interface IDirectionInfoProps extends IWithClassName {
    context: IBusesFilledContext;
    segments: IBusesSeoSegment[] | null;
    minPrice?: number;
    maxPrice?: number;
}

const DirectionInfo: React.FC<IDirectionInfoProps> = props => {
    const {className, context, segments, minPrice, maxPrice} = props;

    if (!segments?.length) {
        return null;
    }

    const departure = context.from.popularTitle || context.from.title;
    const arrival = context.to.popularTitle || context.to.title;

    const title = i18nBlock.directionInfoTitle({
        departure,
        arrival,
    });

    const sortedByDepartureSegments = [...segments].sort((segmentA, segmentB) =>
        parseDate(segmentA.departureTime, ROBOT_DATETIME).isSameOrAfter(
            parseDate(segmentB.departureTime, ROBOT_DATETIME),
        )
            ? 1
            : -1,
    );

    const minDepartureSegment = head(sortedByDepartureSegments);

    if (!minDepartureSegment) {
        return null;
    }

    const minDeparture = formatDate(
        moment(minDepartureSegment.departureTime).tz(
            minDepartureSegment.from.timezone,
        ),
        TIME,
    );
    const minArrival =
        minDepartureSegment.arrivalTime &&
        formatDate(
            moment(minDepartureSegment.arrivalTime).tz(
                minDepartureSegment.to.timezone,
            ),
            TIME,
        );

    const maxDepartureSegment = last(sortedByDepartureSegments);

    if (!maxDepartureSegment) {
        return null;
    }

    const maxDeparture = formatDate(
        moment(maxDepartureSegment.departureTime).tz(
            maxDepartureSegment.from.timezone,
        ),
        TIME,
    );
    const maxArrival =
        maxDepartureSegment.arrivalTime &&
        formatDate(
            moment(maxDepartureSegment.arrivalTime).tz(
                maxDepartureSegment.to.timezone,
            ),
            TIME,
        );

    const minDurationSegment = head(
        [...segments].sort(
            (segmentA, segmentB) =>
                (segmentA.duration || Infinity) -
                (segmentB.duration || Infinity),
        ),
    );

    const duration = minDurationSegment?.duration;

    const description =
        minPrice &&
        maxPrice &&
        insertJSXIntoKey(i18nBlock.directionInfoDescription2)({
            departure,
            arrival,
            departureGenitive: context.from.titleGenitive,
            arrivalAccusative: context.to.titleAccusative,
            arrivalGenitive: context.to.titleGenitive,
            needShowFirstAndLastVariants:
                minDeparture !== maxDeparture &&
                minArrival &&
                maxArrival &&
                duration,
            minPrice: (
                <Price
                    key="minPrice"
                    value={minPrice}
                    currency={CurrencyType.RUB}
                    isRound
                />
            ),
            maxPrice:
                minPrice === maxPrice ? null : (
                    <Price
                        key="maxPrice"
                        value={maxPrice}
                        currency={CurrencyType.RUB}
                        isRound
                    />
                ),
            minDeparture,
            minArrival,
            maxDeparture: minDeparture === maxDeparture ? null : maxDeparture,
            maxArrival,
            duration: duration
                ? i18nDurationBlock
                      .dDashHDashMDashFormat(getDuration(duration))
                      .trim()
                : null,
            distance: context.distance
                ? i18nCommonUnitsBlock.kmDashValue({
                      value: formatInteger(Math.round(context.distance)),
                  })
                : null,
        });

    return (
        <Box className={cx(className)} between={4}>
            <Heading level={2}>{title}</Heading>

            <Text className={cx('description')} size="m" tag="div">
                {description}
            </Text>
        </Box>
    );
};

export default React.memo(DirectionInfo);
