import React from 'react';

import {IBusesFilledContext} from 'types/buses/common/IBusesContext';

import {CurrencyType} from 'utilities/currency/CurrencyType';
import getSearchUrl from 'projects/buses/utilities/urls/getSearchUrl';
import getBusTripSchemaMarkup from 'components/SchemaMarkup/utilities/getBusTripSchemaMarkup';
import {formatPrice} from 'utilities/currency';

import * as i18nBlock from 'i18n/buses-SearchDirectionPage';

import DocumentMeta from 'components/DocumentMeta/DocumentMeta';
import SocialSharingMeta from 'components/SocialSharingMeta/SocialSharingMeta';
import SchemaMarkup, {
    getAggregateOffer,
} from 'components/SchemaMarkup/SchemaMarkup';

interface IHelmetProps {
    context: IBusesFilledContext;
    minPrice: number | null;
    nonce: string;
}

const Helmet: React.FC<IHelmetProps> = props => {
    const {
        context: {from, to},
        minPrice,
        nonce,
    } = props;

    const departure = from.popularTitle || from.title;
    const arrival = to.popularTitle || to.title;

    const canonicalUrl = getSearchUrl({
        fromSlug: from.slug,
        toSlug: to.slug,
        options: {
            withOrigin: true,
        },
    });

    const aggregateOffer = minPrice
        ? getAggregateOffer({
              name: i18nBlock.seoProductName({
                  departure,
                  arrival,
              }),
              price: {
                  value: minPrice,
                  currency: CurrencyType.RUB,
              },
              url: canonicalUrl,
          })
        : null;

    const busTrip = getBusTripSchemaMarkup({
        url: canonicalUrl,
        departure,
        arrival,
    });

    const title = i18nBlock.seoTitle({
        departure,
        arrival,
    });

    const description = i18nBlock.newSeoDescription({
        departure,
        arrival,
        minPrice: minPrice
            ? formatPrice({value: minPrice, currency: CurrencyType.RUB})
            : '',
    });

    return (
        <>
            <DocumentMeta
                title={title}
                description={description}
                canonicalUrl={canonicalUrl}
            />

            <SocialSharingMeta
                url={canonicalUrl}
                title={title}
                description={description}
            />

            {aggregateOffer && (
                <SchemaMarkup data={aggregateOffer} nonce={nonce} />
            )}

            <SchemaMarkup data={busTrip} nonce={nonce} />
        </>
    );
};

export default React.memo(Helmet);
