import React, {useMemo} from 'react';

import {IBusesFilledContext} from 'types/buses/common/IBusesContext';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';

import * as i18nBlock from 'i18n/buses-SearchDirectionPage';

import Heading from 'components/Heading/Heading';
import Box from 'components/Box/Box';
import Text from 'components/Text/Text';
import TextSkeleton from 'components/Skeletons/TextSkeleton/TextSkeleton';

import cx from './TitleWithDescription.scss';

interface ITitleWithDescriptionProps {
    context?: IBusesFilledContext;
    isLoading: boolean;
}

const TitleWithDescription: React.FC<ITitleWithDescriptionProps> = props => {
    const {context, isLoading} = props;

    const deviceType = useDeviceType();

    const content = useMemo(() => {
        if (isLoading || !context) {
            if (deviceType.isMobile) {
                return (
                    <>
                        <Box>
                            <TextSkeleton
                                className={cx('titleSkeletonFirstRow')}
                                size="xl"
                                withAnimation
                            />

                            <TextSkeleton
                                className={cx('titleSkeletonSecondRow')}
                                size="xl"
                                withAnimation
                            />
                        </Box>

                        <Box className={cx('descriptionSkeletonWrapper')}>
                            <TextSkeleton
                                className={cx('descriptionSkeletonFirstRow')}
                                size="m"
                                withAnimation
                            />

                            <TextSkeleton
                                className={cx('descriptionSkeletonSecondRow')}
                                size="m"
                                withAnimation
                            />

                            <TextSkeleton
                                className={cx('descriptionSkeletonThirdRow')}
                                size="m"
                                withAnimation
                            />

                            <TextSkeleton
                                className={cx('descriptionSkeletonFourthRow')}
                                size="m"
                                withAnimation
                            />

                            <TextSkeleton
                                className={cx('descriptionSkeletonFifthRow')}
                                size="m"
                                withAnimation
                            />
                        </Box>
                    </>
                );
            }

            return (
                <>
                    <TextSkeleton
                        className={cx('titleSkeleton')}
                        size="xxl"
                        withAnimation
                    />

                    <Box className={cx('descriptionSkeletonWrapper')}>
                        <TextSkeleton
                            className={cx('descriptionSkeletonFirstRow')}
                            size="m"
                            withAnimation
                        />

                        <TextSkeleton
                            className={cx('descriptionSkeletonSecondRow')}
                            size="m"
                            withAnimation
                        />
                    </Box>
                </>
            );
        }

        const title = i18nBlock.title({
            departure: context.from.popularTitle || context.from.title,
            arrival: context.to.popularTitle || context.to.title,
        });

        const description = i18nBlock.descriptionUnderTitle();

        return (
            <>
                <Heading className={cx('title')} level={1}>
                    {title}
                </Heading>

                <Text className={cx('description')} size="m" tag="div">
                    {description}
                </Text>
            </>
        );
    }, [isLoading, context, deviceType.isMobile]);

    return (
        <Box
            className={cx(deviceMods('root', deviceType))}
            above={10}
            between={deviceType.isMobile ? 2 : 4}
        >
            {content}
        </Box>
    );
};

export default React.memo(TitleWithDescription);
