import React, {useMemo} from 'react';
import moment from 'moment-timezone';

import {IBusesFilledContext} from 'types/buses/common/IBusesContext';
import {
    EBusesCalendar,
    TBusesCalendar,
} from 'types/buses/search/TBusesCalendar';

import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import {HUMAN, ROBOT} from 'utilities/dateUtils/formats';
import getSearchUrl from 'projects/buses/utilities/urls/getSearchUrl';
import {formatDate} from 'utilities/dateUtils';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nSearchPageBlock from 'i18n/buses-SearchPage';
import * as i18nCommonWordLinksBlock from 'i18n/common-word-links';
import * as i18nCommonBlock from 'i18n/common';

import Link from 'components/Link/Link';
import EmptySerp from 'components/EmptySerp/EmptySerp';
import Container from 'components/Layouts/Container/Container';

import cx from './EmptySearch.scss';

interface IEmptySearchProps extends IWithQaAttributes {
    context: IBusesFilledContext;
    /** Для поиска без даты (страница направления) календарь не нужен */
    calendar?: TBusesCalendar;
}

export const DAYS_AROUND_SEARCH = 10;

const MAX_CLOSEST_DATES = 2;

const EmptySearch: React.FC<IEmptySearchProps> = props => {
    const {context, calendar} = props;

    const searchDate = useMemo(
        () => (context.when ? moment(context.when) : null),
        [context.when],
    );

    const closestDates = useMemo(() => {
        if (!searchDate) {
            return [];
        }

        const dates: string[] = [];

        const calendarMap = new Map(Object.entries(calendar || {}));

        let daysAround = 0;

        while (dates.length < MAX_CLOSEST_DATES && calendarMap.size) {
            const datesBeforeAndAfter = [
                moment(searchDate).subtract(daysAround, 'days').format(ROBOT),
                moment(searchDate).add(daysAround, 'days').format(ROBOT),
            ];

            daysAround++;

            for (const differentDate of datesBeforeAndAfter) {
                const dateInfo = calendarMap.get(differentDate);

                if (dateInfo) {
                    if (dateInfo.type === EBusesCalendar.VALID) {
                        dates.push(differentDate);
                    }

                    calendarMap.delete(differentDate);
                }

                if (dates.length === MAX_CLOSEST_DATES) {
                    return dates.sort();
                }
            }
        }

        return dates.sort();
    }, [calendar, searchDate]);

    const text = useMemo(() => {
        if (closestDates.length && searchDate) {
            return insertJSXIntoKey(i18nSearchPageBlock.emptySearchDateText)({
                currentDate: formatDate(searchDate, HUMAN, {withNbsp: true}),
                otherDates: (
                    <span key="otherDates">
                        {closestDates.map((date, index) => {
                            const url = getSearchUrl({
                                fromSlug: context.from.slug,
                                toSlug: context.to.slug,
                                when: moment(date).format(ROBOT),
                            });

                            return (
                                <span key={index}>
                                    {index !== 0 && (
                                        <span>
                                            {' '}
                                            {i18nCommonWordLinksBlock.or()}{' '}
                                        </span>
                                    )}

                                    <Link to={url}>
                                        {formatDate(moment(date), HUMAN, {
                                            withNbsp: true,
                                        })}
                                    </Link>
                                </span>
                            );
                        })}
                    </span>
                ),
            });
        }

        return i18nSearchPageBlock.emptySearchText();
    }, [closestDates, searchDate, context]);

    return (
        <Container className={cx('root')}>
            <EmptySerp
                className={cx('empty')}
                title={i18nSearchPageBlock.emptySearchTitle2({
                    direction: i18nCommonBlock.direction({
                        from: context.from.popularTitle || context.from.title,
                        to: context.to.popularTitle || context.to.title,
                    }),
                })}
                text={text}
                withIcon
                {...prepareQaAttributes(props)}
            />
        </Container>
    );
};

export default React.memo(EmptySearch);
