import {IBusesFilledContext} from 'types/buses/common/IBusesContext';
import {isNotNull} from 'types/utilities';
import {IFilledContextPoint} from 'server/api/RaspApi/types/IRaspParseContextApiResponse';

import getIndexUrl from 'projects/buses/utilities/urls/getIndexUrl';
import getSearchUrl from 'projects/buses/utilities/urls/getSearchUrl';
import getCityPageUrl from 'projects/buses/utilities/urls/getCityUrl';
import getHumanWhen from 'utilities/dateUtils/when/getHumanWhen';

import * as i18nBlock from 'i18n/buses-FooterBreadcrumbs';

interface IBusesCrumb {
    url: string;
    metaUrl: string;
    name: string;
    title: string;
}

const INDEX_PAGE_CRUMB: IBusesCrumb = {
    url: getIndexUrl(),
    metaUrl: getIndexUrl({
        options: {
            withOrigin: true,
        },
    }),
    name: i18nBlock.indexPage(),
    title: i18nBlock.indexPage(),
};

function getDirectionPageCrumb(context: IBusesFilledContext): IBusesCrumb {
    const title = i18nBlock.searchDirectionPage({
        from: context.from.popularTitle || context.from.title,
        to: context.to.popularTitle || context.to.title,
    });

    return {
        url: getSearchUrl({
            fromSlug: context.from.slug,
            toSlug: context.to.slug,
        }),
        metaUrl: getSearchUrl({
            fromSlug: context.from.slug,
            toSlug: context.to.slug,
            options: {
                withOrigin: true,
            },
        }),
        name: title,
        title,
    };
}

function getDatePageCrumb(context: IBusesFilledContext): IBusesCrumb | null {
    if (!context.originWhen) {
        return null;
    }

    const title = i18nBlock.searchDatePage({
        date: getHumanWhen(context.originWhen, {withPossibleYear: true}),
    });

    return {
        url: getSearchUrl({
            fromSlug: context.from.slug,
            toSlug: context.to.slug,
            when: context.originWhen,
        }),
        metaUrl: getSearchUrl({
            fromSlug: context.from.slug,
            toSlug: context.to.slug,
            when: context.originWhen,
            options: {
                withOrigin: true,
            },
        }),
        name: title,
        title,
    };
}

function getCityPageCrumb(city: IFilledContextPoint): IBusesCrumb {
    return {
        url: getCityPageUrl({
            slug: city.slug,
        }),
        metaUrl: getCityPageUrl({
            slug: city.slug,
            options: {
                withOrigin: true,
            },
        }),
        name: city.title,
        title: city.title,
    };
}

export default function getCrumbs(
    context: IBusesFilledContext | undefined,
    showDateItem: boolean,
    city?: IFilledContextPoint,
): IBusesCrumb[] {
    return [
        INDEX_PAGE_CRUMB,
        city ? getCityPageCrumb(city) : null,
        context ? getDirectionPageCrumb(context) : null,
        context && showDateItem ? getDatePageCrumb(context) : null,
    ].filter(isNotNull);
}
