import React, {Fragment, useMemo} from 'react';
import times from 'lodash/times';

import {ADFOX_BANNER_SEARCH_POSITION} from 'constants/AdFox';
import {SUBSCRIPTION_BLOCK_OFFSET} from 'constants/subscription';

import {IBusesSegment} from 'types/buses/search/IBusesSegment';
import {IBusesSearchBookingInfoStore} from 'reducers/buses/types/search/IBusesSearchBookingInfoStore';
import {IBusesFilledContext} from 'types/buses/common/IBusesContext';
import {ESubscriptionVerticalName} from 'types/subscription/ESubscriptionVerticalName';
import {ESubscriptionSource} from 'types/subscription/ESubscriptionSource';
import {EAdFoxBannerPosition, EAdFoxBannerType} from 'types/AdFox';

import {
    getQa,
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {atPlaceOrEarly} from 'utilities/actions/atPlaceOrEarly';

import OptionalSubscription from 'containers/Subscriptions/Inline/OptionalSubscription';

import Box from 'components/Box/Box';
import AdFoxBanner from 'components/AdFoxBanner/AdFoxBanner';
import StationMap from 'projects/buses/components/StationMap/StationMap';
import useStationsMap from 'projects/buses/components/StationMap/hooks/useStationsMap';
import Segment from 'projects/buses/pages/search/components/Segments/components/Segment/Segment';
import SegmentSkeleton from 'projects/buses/pages/search/components/Segments/components/SegmentSkeleton/SegmentSkeleton';
import UpdateNotification from 'projects/buses/pages/search/SearchDatePage/components/UpdateNotification/UpdateNotification';

interface ISegmentsProps extends IWithQaAttributes {
    items: IBusesSegment[] | null;
    hiddenItemsIds?: Set<string> | null;
    context?: IBusesFilledContext | null;
    bookingInfo: IBusesSearchBookingInfoStore;
    isLoading?: boolean;
    advertisingPosition?: number;
    when?: string;
}

const SKELETON_SEGMENTS_COUNT = 4;

const Segments: React.FC<ISegmentsProps> = props => {
    const {
        isLoading = false,
        items,
        hiddenItemsIds,
        context,
        bookingInfo,
        advertisingPosition = ADFOX_BANNER_SEARCH_POSITION,
        when,
    } = props;

    const rootQa = getQa(props);

    const areAllItemsHidden =
        items && hiddenItemsIds && items.length === hiddenItemsIds.size;

    const {
        points: mapPoints,
        setPoints: setMapPoints,
        closeMap,
    } = useStationsMap();

    const content = useMemo(() => {
        if (isLoading || !items) {
            return times(SKELETON_SEGMENTS_COUNT).map(index => (
                <SegmentSkeleton
                    key={index}
                    {...prepareQaAttributes({
                        parent: rootQa,
                        current: 'skeleton',
                        key: index,
                    })}
                />
            ));
        }

        const visibleItemsCount = hiddenItemsIds
            ? items.filter(item => !hiddenItemsIds.has(item.id)).length
            : items.length;

        return (
            <>
                {items.map((item, index) => (
                    <Fragment key={item.id}>
                        <Segment
                            segment={item}
                            context={context}
                            isHidden={Boolean(hiddenItemsIds?.has(item.id))}
                            isLoadingBookingInfo={
                                bookingInfo.isLoading &&
                                bookingInfo.value?.id === item.id
                            }
                            setMapPoints={setMapPoints}
                            when={when}
                            {...prepareQaAttributes({
                                parent: rootQa,
                                current: 'item',
                                key: item.id,
                            })}
                        />

                        {!areAllItemsHidden &&
                            atPlaceOrEarly(
                                visibleItemsCount,
                                index,
                                advertisingPosition,
                            ) && (
                                <AdFoxBanner
                                    type={EAdFoxBannerType.Inline}
                                    position={EAdFoxBannerPosition.Center}
                                />
                            )}

                        {!areAllItemsHidden &&
                            atPlaceOrEarly(
                                visibleItemsCount,
                                index,
                                advertisingPosition + SUBSCRIPTION_BLOCK_OFFSET,
                            ) && (
                                <OptionalSubscription
                                    key="subscription"
                                    vertical={ESubscriptionVerticalName.Bus}
                                    source={ESubscriptionSource.SEARCH}
                                    type="wide"
                                />
                            )}
                    </Fragment>
                ))}

                {context && <UpdateNotification context={context} />}
            </>
        );
    }, [
        isLoading,
        items,
        when,
        hiddenItemsIds,
        context,
        rootQa,
        bookingInfo.isLoading,
        bookingInfo.value?.id,
        setMapPoints,
        areAllItemsHidden,
        advertisingPosition,
    ]);

    return (
        <Box between={3} above={areAllItemsHidden ? 0 : 5}>
            {content}

            <StationMap points={mapPoints} onClose={closeMap} />
        </Box>
    );
};

export default React.memo(Segments);
