import {useMemo} from 'react';
import moment from 'moment-timezone';

import {IBusesSegment} from 'types/buses/search/IBusesSegment';
import {IBusesCommonSegmentInfo} from 'projects/buses/components/Segment/types';
import {IBusesMapPoints} from 'projects/buses/components/StationMap/types/IBusesMapPoints';

import {CurrencyType} from 'utilities/currency/CurrencyType';
import getMapPoint from 'projects/buses/pages/search/components/Segments/components/Segment/utilities/getMapPoint';

import useMap from 'projects/buses/components/Segment/hooks/useMap';

export interface IBusesCommonSegmentInfoParams {
    segment: IBusesSegment;
    bookUrl?: string;
    freeSeatsInfo: string;
    setMapPoints(points: IBusesMapPoints | null): void;
    handleBookClick(): void;
}

// TODO: Почти 1 к 1 src/projects/buses/components/OrderSegment/hooks/useCommonSegmentProps/useCommonSegmentProps.ts
export default function useCommonSegmentInfo({
    segment,
    bookUrl,
    freeSeatsInfo,
    setMapPoints,
    handleBookClick,
}: IBusesCommonSegmentInfoParams): IBusesCommonSegmentInfo {
    const {handleFromClick, handleToClick, handleShowStationsOnMap} = useMap({
        from: getMapPoint(segment.from),
        to: getMapPoint(segment.to),
        setMapPoints,
    });

    return useMemo(
        (): IBusesCommonSegmentInfo => ({
            departure: {
                name: segment.from.title,
                time: moment.tz(segment.departureTime, segment.from.timezone),
                isBoldNameText: true,
                onClick: handleFromClick,
            },
            arrival: {
                name: segment.to.title,
                time: segment.arrivalTime
                    ? moment.tz(segment.arrivalTime, segment.to.timezone)
                    : null,
                isBoldNameText: true,
                onClick: handleToClick,
            },
            additionalInfo: {
                duration: segment.duration,
                carrier: segment.carrierName || segment.carrierCode,
                isSecondaryBusAndCarrierText: true,
            },
            book: {
                price: {
                    currency: CurrencyType.RUB,
                    value: segment.price.amount,
                },
                url: bookUrl,
                freeSeatsInfo,
                onClick: handleBookClick,
            },
            onShowStationsOnMap: handleShowStationsOnMap,
        }),
        [
            bookUrl,
            handleShowStationsOnMap,
            freeSeatsInfo,
            handleBookClick,
            handleFromClick,
            handleToClick,
            segment.arrivalTime,
            segment.carrierCode,
            segment.carrierName,
            segment.departureTime,
            segment.duration,
            segment.from.title,
            segment.from.timezone,
            segment.price.amount,
            segment.to.title,
            segment.to.timezone,
        ],
    );
}
