import moment from 'moment-timezone';

import {IBusesFilledContext} from 'types/buses/common/IBusesContext';
import {IBusesSegment} from 'types/buses/search/IBusesSegment';
import {IBusesGeoPoint} from 'types/buses/search/IBusesGeoPoint';
import {Request} from '@yandex-data-ui/core/lib/types';

import {retry} from 'utilities/async/retry';
import {delay} from 'utilities/async/delay';
import requestGeoPoints from 'projects/buses/utilities/api/requestGeoPoints';
import getPatchedSearchSegments from 'projects/buses/utilities/api/utilities/getPatchedSearchSegments';
import {getNow} from 'utilities/dateUtils';

import {buses as busesProvider} from 'serviceProvider/buses/buses';

const RETRIES_COUNT = 40;
const MAX_SECONDS_DELAY = 3;

export default async function requestSearchSegments({
    req,
    context,
}: {
    req?: Request;
    context: IBusesFilledContext;
}): Promise<IBusesSegment[]> {
    const when = context.when;

    if (!when) {
        return [];
    }

    let triesCounter = 0;

    const segments = await retry(
        async () => {
            await delay(Math.min(triesCounter++, MAX_SECONDS_DELAY - 1) * 1000);

            const {segments: responseSegments, isQuerying} = await busesProvider
                .provider(req?.container)
                .getSegments({
                    fromId: context.from.key,
                    fromTimezone: context.from.timezone,
                    toId: context.to.key,
                    toTimezone: context.to.timezone,
                    date: when,
                });

            if (isQuerying) {
                throw new Error('Сегменты еще не загрузились.');
            }

            return responseSegments;
        },
        {attempts: RETRIES_COUNT, delay: 1000},
    )();

    if (!segments.length) {
        return [];
    }

    const nowMoment = moment(getNow());

    const actualSegments = segments.filter(({departureTime}) =>
        moment(departureTime).isAfter(nowMoment),
    );

    if (!actualSegments.length) {
        return [];
    }

    let geoPoints: IBusesGeoPoint[] = [];

    try {
        geoPoints = await requestGeoPoints({
            req,
            segments: actualSegments,
        });
    } catch (err) {
        /*
         * Ручка получения информации о пунктах отправления/прибытия не должна ломать весь флоу поиска
         */
    }

    return getPatchedSearchSegments(actualSegments, geoPoints, context);
}
