import moment from 'moment';

import {IBusesApiSegment} from 'server/api/BusesApi/types/IBusesGetSegmentsApiResponse';
import {IBusesGeoPoint} from 'types/buses/search/IBusesGeoPoint';
import {IBusesSegment} from 'types/buses/search/IBusesSegment';
import {IBusesFilledContext} from 'types/buses/common/IBusesContext';

import {ROBOT_DATETIME} from 'utilities/dateUtils/formats';

/**
 * Функция добавляет сегментам информацию о пунктах прибытия/отправления и длительность поездки.
 */
export default function getPatchedSearchSegments(
    segments: IBusesApiSegment[],
    geoPoints: IBusesGeoPoint[],
    context: IBusesFilledContext,
): IBusesSegment[] {
    const geoPointsMap = new Map<string, IBusesGeoPoint>();

    geoPoints.forEach(point => geoPointsMap.set(point.code, point));

    return segments.map(segment => {
        const fromPoint = segment.from.id
            ? geoPointsMap.get(`s${segment.from.id}`)
            : null;
        const toPoint = segment.to.id
            ? geoPointsMap.get(`s${segment.to.id}`)
            : null;

        const fromTimezone = fromPoint?.timezone || context.from.timezone;
        const toTimezone = toPoint?.timezone || context.to.timezone;

        console.log('\n\n');
        console.log('departureTime: ' + segment.departureTime);
        console.log('arrivalTime: ' + segment.arrivalTime);

        const patchedSegment: IBusesSegment = {
            ...segment,
            departureTime: moment
                .tz(segment.departureTime, fromTimezone)
                .format(ROBOT_DATETIME),
            arrivalTime: segment.arrivalTime
                ? moment
                      .tz(segment.arrivalTime, toTimezone)
                      .format(ROBOT_DATETIME)
                : null,
            duration: segment.duration || null,
            from: {
                ...segment.from,
                ...(fromPoint || {}),
                timezone: fromTimezone,
                title: fromPoint?.name || segment.fromDesc,
                coordinates:
                    fromPoint?.latitude && fromPoint?.longitude
                        ? fromPoint
                        : undefined,
            },
            to: {
                ...segment.to,
                ...(toPoint || {}),
                timezone: toTimezone,
                title: toPoint?.name || segment.toDesc,
                coordinates:
                    toPoint?.latitude && toPoint.longitude
                        ? toPoint
                        : undefined,
            },
        };

        return patchedSegment;
    });
}
